/**
 * @author Nuno Pereira
 * Polytechnic Institute of Porto 
 * IPP-HURRAY! 2006
 * 
 * This module contains the WiDOM protocol state machine  
 *  
 *  
 */

// nesC includes do not like macros, on the other hand, enums
// do not work with #include...
// So, i had to separate both
includes wiDOM;

////////// TEMP: just for signal if message received won a tournament against us 
includes WiDOMTest;
//////////

#include "widom_macros.h"

module WiDOMM {
        provides {
                interface SplitControl;
                interface BareSendMsg as Send;
                interface ReceiveMsg as Receive;
        }
        uses {
				interface WiDOMCarrierSense;
				interface WiDOMCarrierPulse;
				interface BareSendMsg as RadioSend;
				interface ReceiveMsg as RadioReceive;
				interface SplitControl as WiDOMRadioControl;
				
				interface WiDOMLogicalTime as LogicalTime;
				interface SplitControl as ClockControl;				
	    }
}

implementation {

		//! MAC protocol current state
		uint8_t 	state;
		uint8_t 	prevState;
		
		//! MAC protocol variable: priobits index counter
		uint8_t 	i;
		
		//!	MAC protocol variable: indicates if the node won the tournament
		uint8_t 	winner;
	
		//!	MAC protocol variable: indicates the priority of the message the node is trying to send
		uint16_t 	prio;
		uint8_t 	prio_i;
		
		//!	MAC protocol variable: indicates the prio of the message that won the tournament
		uint16_t 	winner_prio;
		
		//! Message to be/being sent by the protocol (for now, our "message queue") 
        TOS_MsgPtr 	sendMsg;	
                
		//! for now, our "message queue" 
        TOS_MsgPtr 	sendQueue;	
		uint16_t sendQueuePrio;

		//! for temporarily store a message to be signaled as failed
		TOS_MsgPtr 	failSendMsg;
				
		//! Indicate that upper layers can request to send (queue is empty and send message also)
        uint8_t 	canSend;
	
		//!	indicates if we are listenning 
		uint8_t 	listen;
		
		//!	indicates if we make transition 3->5
		uint8_t 	tr3to5;
				
		// some temporary test stats
	  	uint32_t numTournaments=0;
	  	uint32_t numRxTimeouts=0;
	  	uint32_t numTxTimeouts=0;
				
        /************* START OF SPLIT CONTROL INIT FUNCITONS **********/
	
		task void SendDataMsg( );
		void doWiDOMProtocol(uint8_t messageType);
		
        // 
        command result_t SplitControl.init() {
        	atomic {
        		prevState = state = MAC_STATE_0;
				sendMsg = NULL;
				sendQueue = NULL;
				canSend = TRUE;
				listen = TRUE;
        	}
        	call ClockControl.init();
        	return call WiDOMRadioControl.init();
        }

        // 
        command result_t SplitControl.start() {
        	call ClockControl.start();
        	return call WiDOMRadioControl.start();
        }

        // 
        command result_t SplitControl.stop() {
        	call ClockControl.stop();
	        return call WiDOMRadioControl.stop();
        }

		event result_t WiDOMRadioControl.initDone() {
			return signal SplitControl.initDone();
		}

		event result_t WiDOMRadioControl.startDone() {
			signal SplitControl.startDone();
			doWiDOMProtocol(MSG_TYPE_MAC_STATE_ENTER);				
			return SUCCESS;
		}
		
		event result_t WiDOMRadioControl.stopDone() {
			return signal SplitControl.stopDone();
		}
    
		event result_t ClockControl.initDone() {
			return SUCCESS;
		}

		event result_t ClockControl.startDone() {
			return SUCCESS;
		}
		
		event result_t ClockControl.stopDone() {
			return SUCCESS;
		}
	    
        /************* END OF SPLIT CONTROL INIT FUNCITONS **********/

        /**********************************************************
          * WiDOM Protocol
          *	
          **********************************************************/

		task void SendDataMsg( ) {
			atomic {
				call RadioSend.send(sendMsg);
			} 
		}

		task void SignalFailSendMsg( ) {
			atomic {
				signal Send.sendDone(failSendMsg, FAIL);
			} 
		}

		void doWiDOMProtocol(uint8_t messageType) {
			atomic {  
				switch (state) { 
					case MAC_STATE_0: 
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_10 or MAC_STATE_11 
								_setRadioTestMode(); // prepare radio for carrier tx
								_radioSetCarrierSenseOn(); // try to detect a carrier; will always trigger a 1st event signalling the current state 
								_resetTime();															
								_toState(1);		
								break;
							default: 
								//NOP
						}
						break;								
					case MAC_STATE_1:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_0
								// NOP (waiting for carrier sense events)																
								break;
							case MSG_TYPE_RADIO_END_RX: // this is a special case, we had (probably?) previously received a channel busy indication, now channel should be clear
								_cancelAlarm(); // to be safe, cancel any pending alarms
							case MSG_TYPE_RADIO_CHANNEL_IDLE: // channel idle detected							
								_resetTime();
								_setAlarm(TFCS, WAIT_TIME_TFCS); // next state transition when x=TFCS
								break;				
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // channel busy detected
								_cancelAlarm();
								break;
							case MSG_TYPE_MAC_WAIT_TIME_TFCS: // ended waiting for TFCS time units
								_resetTime();
								_toState(2);
								break;
							default:
								// NOP
						}		
						break;
					case MAC_STATE_2:											
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_1	
								_setAlarm(F, WAIT_TIME_F); // continue to sense the medium for a "long time interval"
								break;
//							case MSG_TYPE_RADIO_END_RX:																		
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // channel busy detected (while waiting for a a "long time interval")
								_cancelAlarm();
								_resetTime();
								_toState(1); // go back to state 1 (2->1)
								break;
							case MSG_TYPE_MAC_WAIT_TIME_F: // ended waiting for a "long time interval" (F)																											
								_resetTime();						
								_toState(3);							
								break;
							default: 
								// NOP				
						}		
						break;
					case MAC_STATE_3:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_2
								_setAlarm(E, WAIT_TIME_E);																 
								break;		
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // carrier detected when waiting for E time units, or waiting for someone to start a tournament
								_cancelAlarm();
								_resetTime();
								_toState(5);
								break;									
							case MSG_TYPE_MAC_QUEUE: // application queued a message
							case MSG_TYPE_MAC_WAIT_TIME_E: // signaled at the end of E time units (3->4)
								if (_queueEmpty()==FALSE) {
									_radioSetCarrierSenseOff(); // stop carrier sensing
									_radioSetCarrierOn(); // CarrierOn								
									_toState(4);
								}
								break;
							default: 
								// NOP				
						}		
						break;
					case MAC_STATE_4:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_3
								_setAlarm(E+SWX, WAIT_TIME_SWX); // next state transition when x=E+SWX (+SWX time units)
								break;		
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // ignore
								// NOP
								break;													
							case MSG_TYPE_MAC_WAIT_TIME_SWX: // signaled at the end of E time units (4->5)
								_resetTime();
								_toState(5); // (4->5)
								break;
							default: 
								// NOP				
						}		
						break;
					case MAC_STATE_5:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_4														
								_setAlarm(H, WAIT_TIME_H); // next state transition when x=H														 						
								break;
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // ignore
							case MSG_TYPE_RADIO_CHANNEL_IDLE: // ignore
								// NOP
								break;
							case MSG_TYPE_MAC_WAIT_TIME_H: // signaled at the end of H time units 
								if (prevState == MAC_STATE_4) _radioSetCarrierOff(); // carrierOff (check if we made transition 3->5 in wich case, the carrier is off)
								else _radioSetCarrierSenseOff();
								i=0;
								if (_queueEmpty()==FALSE) { // we have something to send, so we init for the tournament
									winner=TRUE;
									listen=FALSE;
									_dequeueMsg(); // dequeue highest prio msg; set prio=message prio									
								} else { // we have nothing to send 
									prio = INVALID_PRIO;
									winner=FALSE;
									listen=TRUE;
									sendMsg=NULL;								
								}						
								_toState(6);
								break;
							default: 
								// NOP				
						}		
						break;					
					case MAC_STATE_6:
						switch (messageType) {				
							case MSG_TYPE_MAC_STATE_ENTER: // enter from (call _toState()) MAC_STATE_5 or MAC_STATE_8																					
								_setAlarm(H+G+(G+H)*i, WAIT_TIME_G);
								break; 
							// while waiting for the guard time, we ignore other messages from radio
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // ignore
							case MSG_TYPE_RADIO_CHANNEL_IDLE: // ignore
								// NOP
								break;							
							case MSG_TYPE_MAC_WAIT_TIME_G:
								prio_i = _prio(i);														
								if (prio_i == 0 && winner == TRUE) { // (7->8)
									_winnerPrio(i, 0);
									_radioSetCarrierOn();
								} else {
									_winnerPrio(i, 1);
									_radioSetCarrierSenseOn();
								}
								_toState(7);
								break;
							default:
	       						// NOP
						}
						break;
					case MAC_STATE_7:
						switch (messageType) {				
							case MSG_TYPE_MAC_STATE_ENTER:
								_setAlarm(2*H+G+(G+H)*i, WAIT_TIME_H); // next state transition when x=2*H+G+(G+H)*i (+H time units)
								break;
							case MSG_TYPE_RADIO_CHANNEL_BUSY: // heard a carrier during the tournament; lost the tournament
								  _winnerPrio(i, 1);
								  winner=FALSE;
								break;
							case MSG_TYPE_RADIO_CHANNEL_IDLE: // ignore
								// NOP
								break;														
							case MSG_TYPE_MAC_WAIT_TIME_H: // ended waiting H time units; check results of this tournament round
							    _toState(8);
								break;
							default:
	       						// NOP
						}
						break;
					case MAC_STATE_8:
	   					switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER:
								if (i<NPRIOBITS-1) { // go to the next tournament round
									if (prio_i == 0 && winner == TRUE) {
										_radioSetCarrierOff();
									} else _radioSetCarrierSenseOff();
									i++;
									_toState(6);
								} else if (i==NPRIOBITS-1) { // this was the last tournament round
									if (winner == TRUE) { // we won the tournament																		
										_radioSetNormalMode(); // set radio to "normal" mode
									    _toState(9);
									} else { // if (winner == FALSE) 
										if (listen==FALSE) { // lost tournament										
											_enqueueMsg(); // put message back in the message queue; prio = invalid priority
										}								
										_radioSetNormalMode(); // set radio to "normal" mode (and receive)
										_setAlarm(2*H+G+(G+H)*(NPRIOBITS-1)+ETG+CMAX, WAIT_TIME_RXTIMEOUT); // next state transition
										_toState(11); // (8->11)
									}
								}
								break;
							case MSG_TYPE_RADIO_CHANNEL_IDLE: // ignore
								// NOP
								break;														
							default:
	       						// NOP
						}
						break;
					case MAC_STATE_9:
	   					switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER:
	   							_setAlarm(2*H+G+(G+H)*(NPRIOBITS-1) + ETG, WAIT_TIME_ETG); // next state transition
	   							break;
							// while waiting for the guard time, we ignore other messages from radio
							case MSG_TYPE_RADIO_CHANNEL_BUSY:
							case MSG_TYPE_RADIO_CHANNEL_IDLE:
								// NOP
								break;
							case MSG_TYPE_MAC_WAIT_TIME_ETG: // ended waiting after winning the tournament
								if (post SendDataMsg() == SUCCESS) {
									_toState(10); // Tx message 
								} else { 
									failSendMsg = sendMsg;
									post SignalFailSendMsg();
									sendMsg = NULL;
									if (_queueEmpty() == TRUE) canSend = TRUE;
									_toState(0); // go back to state 0						
								}   							   							
								break;
							default:
								// NOP
						}
						break;					
					case MAC_STATE_10:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER:
								// NOP (wating for end tx)
								break;
							case MSG_TYPE_RADIO_END_TX: // ended sending message
/******************************************************	
 *Sending continuously	(for testing purposes) */
//								sendQueue = sendMsg;
/******************************************************/	
								sendMsg = NULL;
								if (_queueEmpty() == TRUE) canSend = TRUE;
								
															
								_toState(0); // go back to state 0
								break;
							default:
	       						// NOP
	       				}
						break;
					case MAC_STATE_11:
						switch (messageType) {
							case MSG_TYPE_MAC_STATE_ENTER:
								// NOP (wating for end rx)
								break;
							case MSG_TYPE_MAC_WAIT_TIME_RXTIMEOUT: // reception has timed out
								numRxTimeouts++;
								_toState(0);
								break;
							// ignore other messages from radio
							case MSG_TYPE_RADIO_CHANNEL_BUSY:
							case MSG_TYPE_RADIO_CHANNEL_IDLE:
								// NOP
								break;
							case MSG_TYPE_RADIO_END_RX: // ended receiving message
								_cancelAlarm();
								_toState(0); // go back to state 0
								break;
							default:
	       						// NOP
						}
						break;
				}
			} // atomic
		} //doWiDOMProtocol

        /**********************************************************
          * Send
          *	
          **********************************************************/
        command result_t Send.send(TOS_MsgPtr pMsg) {
        	WiDOMTestMsgPtr wiMsgPtr = (WiDOMTestMsgPtr) &pMsg->data[0];
			uint16_t _state;
			bool _canSend;

			atomic _canSend = canSend;

			if ( _canSend == TRUE ) { 
		        atomic {
		        	sendQueue = pMsg;
		        	sendQueuePrio = wiMsgPtr->prio;
					canSend = FALSE;
					_state = state;
				}
				// if protocol is waiting for a message, signal it
				if (_state == MAC_STATE_3) doWiDOMProtocol(MSG_TYPE_MAC_QUEUE);
				return SUCCESS;				
			}
			return FAIL;
        }

		event result_t RadioSend.sendDone(TOS_MsgPtr msg, result_t success) {
			doWiDOMProtocol(MSG_TYPE_RADIO_END_TX);
/******************************************************	
 *Sending continuously	(for testing purposes) */
//			return SUCCESS;
/*******************************************************/	
			return signal Send.sendDone(msg, success);
		}

        /**********************************************************
          * Receive
          *	
          **********************************************************/
		event TOS_MsgPtr RadioReceive.receive(TOS_MsgPtr m) {
			WiDOMTestMsgRcvPtr wiTestMsg = (WiDOMTestMsgRcvPtr) &m->data[0];

			// check if message received won a tournament against us
			// indicate that in packet's payload
			atomic {
				if (listen == FALSE && winner == FALSE) wiTestMsg->won_tournament = 1;
				else wiTestMsg->won_tournament = 0;

				if (m->type == WiDOMTEST_PKT_TYPE_FINISH) { // packet signalling the end (fill packet with test statistics)
		  			wiTestMsg->numTournaments = numTournaments-1;
		  			wiTestMsg->numRxTimeouts = numRxTimeouts;
		  			wiTestMsg->numTxTimeouts = numTxTimeouts;
				}
			}

			doWiDOMProtocol(MSG_TYPE_RADIO_END_RX);
			
			// only indicate packet, if it has a correct crc
			if (m->crc == 1) return signal Receive.receive(m);
			else return m;
		}
		
		async event void LogicalTime.alarm(uint8_t type) {
			doWiDOMProtocol(type);
		}	

		async event result_t WiDOMCarrierSense.channelBusy() {
			doWiDOMProtocol(MSG_TYPE_RADIO_CHANNEL_BUSY);
			return SUCCESS;			
		}

		async event result_t WiDOMCarrierSense.channelIdle() {
			doWiDOMProtocol(MSG_TYPE_RADIO_CHANNEL_IDLE);
			return SUCCESS;			
		}
}
