/**
 * Basic application to send packets periodically, using WiDOM.
 *
 * @author Nuno Pereira
 * Polytechnic Institute of Porto 
 * IPP-HURRAY! 2006
 **/

includes WiDOMTest;

// the "finisher" only hears txs and ends experiment
//#define FINISHER

// after how many txs heard the "finisher" will send a "finish" packet
#define END_NUMTXS 51000

#define NUM_COUNTERS 20

module WiDOMTestM {
  provides {
    interface StdControl;
  }
  uses {
  	interface Timer as StatsTimer;
	interface Timer as SendTimer;

    interface Leds;
    
	interface BareSendMsg as Send;
	interface ReceiveMsg as Receive;
	interface SplitControl as WiDOMControl;
			
	interface Random;
	
	interface UARTDebugger;
  }
}
implementation {
	// Two modes: Normal mode to send packets, and stats mode where we send the data collected to the UART
	enum { MODE_NORMAL, MODE_STATS };

	TOS_Msg packetBuffer;		// "Normal" data messages

	TOS_MsgPtr packetSend;

	int8_t mode;
	
	bool msgPending;
	
	uint16_t myPrio;

	uint32_t sndSequenceCounter;
	uint32_t sndMsgsErrorCounter;
	
	uint32_t rcvPackets[NUM_COUNTERS];
	uint32_t rcvCorrectPackets[NUM_COUNTERS];
	uint32_t rcvSequenceCounter[NUM_COUNTERS];
	uint32_t rcvSequenceCounterError[NUM_COUNTERS];

	uint32_t totalRcvMsgsCounter;
	uint32_t totalSeqErrorsCounter;
	uint32_t totalLostTournamentsCounter;
				
	norace uint8_t pendingCount;

	// some temporary stats from MAC
  	uint32_t numMACTournaments;
  	uint32_t numMACRxTimeouts;
  	uint32_t numMACTxTimeouts;
	 
	/**
	 * Initialize the component.
	 * 
	 * @return Always returns <code>SUCCESS</code>
	 **/
	command result_t StdControl.init() {
		call Leds.init();
		call WiDOMControl.init();
		
		call UARTDebugger.init();
		
		return call Random.init();
	}

	task void sendPacket();

	/**
	 * Start things up.  This just sets the rate for the clock component.
	 * 
	 * @return Always returns <code>SUCCESS</code>
	 **/
	command result_t StdControl.start() {
		uint8_t _mode, i;
		
		atomic {
			mode = MODE_NORMAL;
			packetBuffer.length = 0;
			sndSequenceCounter = 1;
			msgPending = FALSE;			
			for (i=0; i<NUM_COUNTERS; i++) {
				rcvPackets[i] = 0;
				rcvCorrectPackets[i] = 0;
				rcvSequenceCounter[i] = 0;
				rcvSequenceCounterError[i] = 0;
			}
			totalRcvMsgsCounter=0;
			totalSeqErrorsCounter=0;
			totalLostTournamentsCounter = 0;
		}

		call WiDOMControl.start();

		myPrio = /*1008 +*/ (TOS_LOCAL_ADDRESS/**2*/);
		call UARTDebugger.sendTaggedIntNL("TOS_LOCAL_ADDRESS=", TOS_LOCAL_ADDRESS );
		call UARTDebugger.sendTaggedIntNL("PRIO=", myPrio );

		atomic _mode = mode;
#ifndef FINISHER		
		if (_mode == MODE_NORMAL) call SendTimer.start(TIMER_ONE_SHOT, 1000);
#else 
//		post sendPacket();
//		call SendTimer.start(TIMER_ONE_SHOT, 1000);
#endif

		return SUCCESS;
	}

	/**
	 * Halt execution of the application.
	 * This just disables the clock component.
	 * 
	 * @return Always returns <code>SUCCESS</code>
	 **/
	command result_t StdControl.stop() {
		call WiDOMControl.stop();
		call SendTimer.stop();
		return SUCCESS;
	}
	
	event result_t WiDOMControl.initDone() {
		return SUCCESS;
	}

	event result_t WiDOMControl.startDone() {
				
		return SUCCESS;
	}

	event result_t WiDOMControl.stopDone() {
		return SUCCESS;
	}

	task void sendPacket() {
		uint8_t i; 
		WiDOMTestMsgPtr wiMsgPtr;
		
		atomic {
			if (msgPending == FALSE) {
				packetBuffer.addr = TOS_BCAST_ADDR;
#ifndef FINISHER				
				packetBuffer.type = WiDOMTEST_PKT_TYPE_DATA;
#else				
				packetBuffer.type = WiDOMTEST_PKT_TYPE_FINISH;
#endif
				packetBuffer.group = TOS_DEFAULT_AM_GROUP;
				packetBuffer.length = WiDOMTEST_MSG_LENGHT;
				packetBuffer.crc = 1;
				// WiDOMTest data
				wiMsgPtr = (WiDOMTestMsgPtr) &packetBuffer.data[0];
				wiMsgPtr->srcAddr = TOS_LOCAL_ADDRESS;
				wiMsgPtr->prio = myPrio;
				wiMsgPtr->seqCounter = sndSequenceCounter;
				wiMsgPtr->won_tournament = 0xAA; // this is not relevant here
				// Put some data in the WiDOMTest payload
				for (i=0; i<WiDOMTEST_DATA_LENGHT; i++) {
					wiMsgPtr->data[i] = i; 
				}
				packetSend = &packetBuffer;
				if ((call Send.send(packetSend)) == SUCCESS) {
					msgPending = TRUE;
				}
			}
		}
	}
	
	event result_t StatsTimer.fired() {
		uint8_t _mode, i;		

		atomic _mode = mode;
				
		if (_mode == MODE_NORMAL) {
			// check what?
		} else {
			// output stats
			call Leds.greenToggle();
			call Leds.redToggle();
			call Leds.yellowToggle();

			call UARTDebugger.sendTaggedIntNL("A=", TOS_LOCAL_ADDRESS);
			call UARTDebugger.sendTaggedInt("TSndM(", (sndSequenceCounter-1)/0xFFFF);
			call UARTDebugger.sendTaggedIntNL(")=", sndSequenceCounter/*-1+1*/ );
			call UARTDebugger.sendTaggedInt("TRcvM(", totalRcvMsgsCounter/0xFFFF);
			call UARTDebugger.sendTaggedIntNL(")=", totalRcvMsgsCounter+1);
			if (totalSeqErrorsCounter < 0xFFFF) {
			    call UARTDebugger.sendTaggedIntNL("TSeqE=", totalSeqErrorsCounter);
			} else {
				call UARTDebugger.sendTaggedInt("TSeqE(", totalSeqErrorsCounter/0xFFFF);
				call UARTDebugger.sendTaggedIntNL(")=", totalSeqErrorsCounter+1);
			}
			call UARTDebugger.sendNL();
			if (numMACTournaments < 0xFFFF) {
			    call UARTDebugger.sendTaggedIntNL("TTrts=", numMACTournaments);
			} else {
				call UARTDebugger.sendTaggedInt("TTrts(", numMACTournaments/0xFFFF);
				call UARTDebugger.sendTaggedIntNL(")=", numMACTournaments+1);
			}
			if (totalLostTournamentsCounter < 0xFFFF) {
			    call UARTDebugger.sendTaggedIntNL("LTrts=", totalLostTournamentsCounter);
			} else {
  				call UARTDebugger.sendTaggedInt("LTrts(", totalLostTournamentsCounter/0xFFFF);
  				call UARTDebugger.sendTaggedIntNL(")=", totalLostTournamentsCounter+1);
			}
			if (numMACRxTimeouts < 0xFFFF) {
			    call UARTDebugger.sendTaggedIntNL("RxTts=", numMACRxTimeouts);
			} else {
  				call UARTDebugger.sendTaggedInt("RxTts(", numMACRxTimeouts/0xFFFF);
  				call UARTDebugger.sendTaggedIntNL(")=", numMACRxTimeouts+1);
			}
			if (numMACTxTimeouts < 0xFFFF) {
			    call UARTDebugger.sendTaggedIntNL("TxTts=", numMACTxTimeouts);
			} else {
  				call UARTDebugger.sendTaggedInt("TxTts(", numMACTxTimeouts/0xFFFF);
  				call UARTDebugger.sendTaggedIntNL(")=", numMACTxTimeouts+1);
			}
			call UARTDebugger.sendNL();
			call UARTDebugger.sendStringNL("A\tTRM\tTPOK\tSErr");
			for (i=0; i<NUM_COUNTERS; i++) {
				if (rcvPackets[i] > 0) {
					call UARTDebugger.sendInt(i);
					if (rcvPackets[i] < 0xFFFF) {
						call UARTDebugger.sendTaggedInt("\t", rcvPackets[i]);
					} else {
						call UARTDebugger.sendTaggedInt("\t(", rcvPackets[i]/0xFFFF);
						call UARTDebugger.sendTaggedInt(")", rcvPackets[i]+1);
					}
					if (rcvPackets[i] == rcvCorrectPackets[i]) call UARTDebugger.sendString("\t==");
					else {
					    if (rcvCorrectPackets[i] < 0xFFFF) {
							call UARTDebugger.sendTaggedInt("\t", rcvCorrectPackets[i]);
					    } else {
							call UARTDebugger.sendTaggedInt("\t(", rcvCorrectPackets[i]/0xFFFF);
							call UARTDebugger.sendTaggedInt(")", rcvCorrectPackets[i]+1);
						}
					}
    				if (rcvSequenceCounterError[i]<0xFFFF) {
    				    call UARTDebugger.sendTaggedInt("\t", rcvSequenceCounterError[i]);
    				} else {
						call UARTDebugger.sendTaggedInt("\t(", rcvSequenceCounterError[i]/0xFFFF);
						call UARTDebugger.sendTaggedInt(")", rcvSequenceCounterError[i]+1);
					}
					call UARTDebugger.sendNL();
				}
			}
		}
		call UARTDebugger.sendNL();
		call UARTDebugger.sendNL();
		return SUCCESS;		
	}
	
	event result_t SendTimer.fired() {
		uint8_t _mode;
		uint16_t nextTime = 0 + (call Random.rand() & 0x1FF); // [0, 511] ms

		atomic _mode = mode;
				
		if (_mode == MODE_NORMAL) {
			post sendPacket();

			// post next send; we can do inside this if, because we never get back from stats mode
			call SendTimer.start(TIMER_ONE_SHOT, nextTime);	
		}	
		
		return SUCCESS;		
	}

	event result_t Send.sendDone(TOS_MsgPtr msg, result_t success) {
		if (success == SUCCESS) {
			atomic {
				if (mode != MODE_STATS) sndSequenceCounter++;
				msgPending = FALSE;
			}
			call Leds.greenToggle();
		} else atomic msgPending = FALSE;		
		
		return SUCCESS;
	}

	event TOS_MsgPtr Receive.receive(TOS_MsgPtr packet) {
#ifndef FINISHER		
		WiDOMTestMsgRcvPtr wiTestMsg = (WiDOMTestMsgRcvPtr) &packet->data[0];
		bool wantedTx; 
		uint32_t *ourCounter, *ourErrorCounter;
#endif		
		uint8_t _mode;
		
		atomic _mode =  mode;			
		
		if (_mode == MODE_STATS) return packet; 

		if (packet->type == WiDOMTEST_PKT_TYPE_DATA) { // "Normal" packet	

#ifdef FINISHER		

			call Leds.yellowToggle();
			
			atomic totalRcvMsgsCounter += 1;

			if (totalRcvMsgsCounter > END_NUMTXS) {
				post sendPacket();
				
				call Leds.greenOn();
				call Leds.redOn();
				call Leds.yellowOn();
			}

#else		

			// first, check on the address received
			if (wiTestMsg->srcAddr >= NUM_COUNTERS)  return packet;

			atomic {
				totalRcvMsgsCounter += 1;
				rcvPackets[wiTestMsg->srcAddr]++;
				wantedTx = (msgPending == TRUE && wiTestMsg->won_tournament == 1);
			}

			if (wantedTx == TRUE) atomic totalLostTournamentsCounter++;

			// if message received won a tournament against us, won_tournament is 1
			if ( (wiTestMsg->prio < myPrio && wantedTx == TRUE) || (wantedTx == FALSE) ) {
				atomic rcvCorrectPackets[wiTestMsg->srcAddr]++;
				call Leds.yellowToggle();
			}

			// pointers to the counters we are going to change
			ourCounter = &rcvSequenceCounter[wiTestMsg->srcAddr];
			ourErrorCounter = &rcvSequenceCounterError[wiTestMsg->srcAddr];			

			atomic {
				if ( wiTestMsg->seqCounter != (*ourCounter) && (*ourCounter) != 0 ) {
					call UARTDebugger.sendTaggedInt("SRC=",wiTestMsg->srcAddr);
					call UARTDebugger.sendTaggedInt(" OUR_SC=", (*ourCounter));					
					call UARTDebugger.sendTaggedIntNL(" PKT_SC=", wiTestMsg->seqCounter);
					call Leds.redToggle();
					(*ourErrorCounter) += (wiTestMsg->seqCounter - (*ourCounter));
					totalSeqErrorsCounter += (wiTestMsg->seqCounter - (*ourCounter));
				}
				(*ourCounter) = wiTestMsg->seqCounter;
				(*ourCounter) += 1;
				
			}
		} if (packet->type == WiDOMTEST_PKT_TYPE_FINISH) { // packet signalling the end
			atomic mode = MODE_STATS;
			
			numMACTournaments = wiTestMsg->numTournaments;
  			numMACRxTimeouts = wiTestMsg->numRxTimeouts;
  			numMACTxTimeouts = wiTestMsg->numTxTimeouts;

			call StatsTimer.start(TIMER_REPEAT, 3000);
#endif								
		}	
		return packet; 
	}
}
