/*									
 * "Copyright (c) 2000-2003 The Regents of the University  of California.  
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the author appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Copyright (c) 2002-2003 Intel Corporation
 * All rights reserved.
 *
 * This file is distributed under the terms in the attached INTEL-LICENSE     
 * file. If you do not find these files, copies can be found by writing to
 * Intel Research Berkeley, 2150 Shattuck Avenue, Suite 1300, Berkeley, CA, 
 * 94704.  Attention:  Intel License Inquiry.
 */

/**
 * @author Nuno Pereira
 * Polytechnic Institute of Porto 
 * IPP-HURRAY! 2006
 * 
 * This module contains bare send/receive functionality used by WiDOM MAC.
 * It can be used directly to send radio messages with no medium access control  
 *  
 * Heavily based on CC2420Radio v1.42, by Joe Polastre and Alan Broad
 * $Id: CC2420RadioM.nc,v 1.42 2005/08/15 18:04:55 jpolastre Exp $ 
 *  
 */
 
includes byteorder;

#define INVALID_CCA_STATE 0xFF

module WiDOMRadioM {
        provides {
                interface StdControl;
                interface SplitControl;
                interface WiDOMCarrierSense;
                interface WiDOMCarrierPulse;
				interface BareSendMsg as Send;
                interface ReceiveMsg as Receive;
                interface RadioCoordinator as RadioSendCoordinator;
                interface RadioCoordinator as RadioReceiveCoordinator;
        }
        uses {
                interface SplitControl as CC2420SplitControl;
                interface CC2420Control;
                interface HPLCC2420 as HPLChipcon;
                interface HPLCC2420FIFO as HPLChipconFIFO;
                interface HPLCC2420Interrupt as FIFOP;
                interface HPLCC2420Capture as SFD;
				interface WiDOMClockTicks as ClockTicks;
                interface Random;
        }
}

implementation {
			
        enum {
                DISABLED_STATE = 0,
                DISABLED_STATE_STARTTASK,
                IDLE_STATE,
                TX_STATE,
                TX_WAIT,
                PRE_TX_STATE,
                POST_TX_STATE,
                POST_TX_ACK_STATE,
                RX_STATE,
                POWER_DOWN_STATE,
                WARMUP_STATE,
				CARRIER_TX_STATE             
        };

        uint8_t stateRadio;
        norace uint8_t currentDSN;
        bool bPacketReceiving;
        uint8_t txlength;
        norace TOS_MsgPtr txbufptr;	// pointer to transmit buffer
        norace TOS_MsgPtr rxbufptr;	// pointer to receive buffer
        TOS_Msg RxBuf;				// save received messages

		bool CCA;
		uint8_t CCAValidTime;
		uint8_t CCAState;
		bool taskCheckCCA; 
								
		void sendEnd() {
        	call SFD.disable();
        	atomic {
        		atomic stateRadio = IDLE_STATE;
            	txbufptr->length = txbufptr->length - MSG_HEADER_SIZE - MSG_FOOTER_SIZE;
            }
		}

        task void SendFailed() {
	        sendEnd();
	        signal Send.sendDone(txbufptr, FAIL);
        }
    
        task void PacketSent() {
	        TOS_MsgPtr pBuf; //store buf on stack
	
			sendEnd();
	        atomic pBuf = txbufptr;
	        signal Send.sendDone(pBuf,SUCCESS);
        }
            
        /***************************************************************************
         * PacketRcvd
         * - Radio packet rcvd, signal 
         ***************************************************************************/
        task void PacketRcvd() {
	        TOS_MsgPtr pBuf;
			
	        atomic {
	                pBuf = rxbufptr;
	        }
	       	       
        	pBuf = signal Receive.receive((TOS_MsgPtr)pBuf);
	        atomic {
                if (pBuf) rxbufptr = pBuf;
                rxbufptr->length = 0;
                bPacketReceiving = FALSE;
	        }
        }
    
        //**********************************************************
        //* Exported interface functions for Std/SplitControl
        //* StdControl is deprecated, use SplitControl
        //**********************************************************/

        // This interface is depricated, please use SplitControl instead
        command result_t StdControl.init() {
	        return call SplitControl.init();
        }

        // Split-phase initialization of the radio
        command result_t SplitControl.init() {

	        atomic {
	                stateRadio = DISABLED_STATE;
	                currentDSN = 0;
	                bPacketReceiving = FALSE;
	                rxbufptr = &RxBuf;
	                rxbufptr->length = 0;
	                //csClkTks = 0;
	                CCA = FALSE;
	                CCAValidTime = 0;
	        }

	        return call CC2420SplitControl.init();
        }

        event result_t CC2420SplitControl.initDone() {
	        return signal SplitControl.initDone();
        }

        default event result_t SplitControl.initDone() {
	        return SUCCESS;
        }

        // This interface is depricated, please use SplitControl instead
        command result_t StdControl.stop() {
	        return call SplitControl.stop();
        }

        // split phase stop of the radio stack
        command result_t SplitControl.stop() {
	        atomic stateRadio = DISABLED_STATE;

	        call SFD.disable();
	        call FIFOP.disable();
	        return call CC2420SplitControl.stop();
        }

        event result_t CC2420SplitControl.stopDone() {
			return signal SplitControl.stopDone();
        }

        default event result_t SplitControl.stopDone() {
	        return SUCCESS;
        }

        task void startRadio() {
	        result_t success = FAIL;
	        atomic {
	                if (stateRadio == DISABLED_STATE_STARTTASK) {
	                stateRadio = DISABLED_STATE;
	                success = SUCCESS;
                }
        }

        if (success == SUCCESS)
		        call SplitControl.start();
        }

        // This interface is depricated, please use SplitControl instead
        command result_t StdControl.start() {
	        // if we put starting the radio from StdControl in a task, then it
	        // delays executing until the other "start" functions are done.
	        // the bug occurs when other components use the underlying bus in their
	        // start() functions.  since the radio is split phase, it acquires
	        // the bus during SplitControl.start() but doesn't release it until
	        // SplitControl.startDone().  Ideally, Main would be changed to
	        // understand SplitControl and run each SplitControl serially.
	        result_t success = FAIL;

	        atomic {
	                if (stateRadio == DISABLED_STATE) {
	                // only allow the task to be posted once.
	                if (post startRadio()) {
			                success = SUCCESS;
			                stateRadio = DISABLED_STATE_STARTTASK;
		                }
	                }
	        }

	        return success;
        }

        // split phase start of the radio stack (wait for oscillator to start)
        command result_t SplitControl.start() {
	        uint8_t chkstateRadio;

	        atomic chkstateRadio = stateRadio;

	        if (chkstateRadio == DISABLED_STATE) {
		        atomic {
		                stateRadio = WARMUP_STATE;
		                rxbufptr->length = 0;
		        }
		        return call CC2420SplitControl.start();
	        }
	        return FAIL;
        }

        event result_t CC2420SplitControl.startDone() {
	        uint8_t chkstateRadio;

	        atomic {
				chkstateRadio = stateRadio;
			}

	        if (chkstateRadio == WARMUP_STATE) {        
		        call CC2420Control.RxMode();

		        //enable interrupt when pkt rcvd
		        call FIFOP.startWait(FALSE);

		        // enable start of frame delimiter timer capture (timestamping)
		        call SFD.enableCapture(TRUE);

		        atomic stateRadio	 = IDLE_STATE;
	        }

	        signal SplitControl.startDone();
        	        	        
	        return SUCCESS;
        }

        default event result_t SplitControl.startDone() {
	        return SUCCESS;
        }

        /************* END OF STDCONTROL/SPLITCONTROL INIT FUNCITONS **********/

        void flushRXFIFO() {
	        call FIFOP.disable();
	        call HPLChipcon.read(CC2420_RXFIFO);	   //flush Rx fifo
	        call HPLChipcon.cmd(CC2420_SFLUSHRX);
	        call HPLChipcon.cmd(CC2420_SFLUSHRX);
	        atomic bPacketReceiving = FALSE;
	        call FIFOP.startWait(FALSE);
        }

        /**
         * send a packet.  
         **/
        void sendPacket() {
	        uint8_t currentstate;

	        atomic currentstate = stateRadio;
	        
			if (currentstate == PRE_TX_STATE) {
/*	
		        // if a FIFO overflow occurs or if the data length is invalid, flush
		        // the RXFIFO to get back to a normal state.
		        if ((!TOSH_READ_CC_FIFO_PIN() && !TOSH_READ_CC_FIFOP_PIN())) {
			        flushRXFIFO();
		        }
*/					
				// start sending now
		        atomic stateRadio = TX_STATE;
		        call HPLChipcon.cmd(CC2420_STXON); // tell radio to start sending, unconditionally

		        // wait for the SFD to go high for the transmit SFD
		        call SFD.enableCapture(TRUE);
		    }
        }

        /**
         * Captured an edge transition on the SFD pin
         * Useful for time synchronization as well as determining
         * when a packet has finished transmission
         */
        async event result_t SFD.captured(uint16_t time) {
	        switch (stateRadio) {
		        case TX_STATE:
		        
			        // wait for SFD to fall--indicates end of packet
			        call SFD.enableCapture(FALSE);
			        // if the pin already fell, disable the capture and let the next
			        // state enable the cpature (bug fix from Phil Buonadonna)
			        if (!TOSH_READ_CC_SFD_PIN()) {
				        call SFD.disable();
			        }
			        else {
				        stateRadio = TX_WAIT;
			        }		        
			        // fire TX SFD event
			        txbufptr->time = time;
			        signal RadioSendCoordinator.startSymbol(8,0,txbufptr);
			        // if the pin hasn't fallen, break out and wait for the interrupt
			        // if it fell, continue on the to the TX_WAIT state
			        if (stateRadio == TX_WAIT) {
				        break;
			        }
		        case TX_WAIT:
			        // end of packet reached
			        stateRadio = POST_TX_STATE;
			        call SFD.disable();
			        // revert to receive SFD capture
			        call SFD.enableCapture(TRUE);
	        		// signal we are done sending this packet
	        		if (!post PacketSent()) post SendFailed();

			        break;
		        default:
			        // fire RX SFD handler
			        rxbufptr->time = time;
			        signal RadioReceiveCoordinator.startSymbol(8,0,rxbufptr);
	        }
	        return SUCCESS;
        }

        /**
         * Start sending the packet data to the TXFIFO of the CC2420
         */
        task void startSend() {
	        // flush the tx fifo of stale data
	        if (!(call HPLChipcon.cmd(CC2420_SFLUSHTX))) {
		        post SendFailed();
		        return;
	        }
	       
	        // write the txbuf data to the TXFIFO
	        if (!(call HPLChipconFIFO.writeTXFIFO(txlength+1,(uint8_t*)txbufptr))) {
		        post SendFailed();
		        return;
	        }
        }

        /**********************************************************
          * Send
          * - Xmit a packet
          *	USE SFD FALLING FOR END OF XMIT !!!!!!!!!!!!!!!!!! interrupt???
          * - If !TxBusy then 
          *   a) Flush the tx fifo 
          *   b) Write Txfifo address
          *	
          **********************************************************/
        command result_t Send.send(TOS_MsgPtr pMsg) {
	        uint8_t currentstate;
	        atomic currentstate = stateRadio;
						
	        if (currentstate == IDLE_STATE) {	
		        // put default FCF values in to get address checking to pass
		        pMsg->fcflo = CC2420_DEF_FCF_LO;
		        pMsg->fcfhi = CC2420_DEF_FCF_HI;
		        // destination PAN is broadcast
		        pMsg->destpan = TOS_BCAST_ADDR;
		        // adjust the destination address to be in the right byte order
		        pMsg->addr = toLSB16(pMsg->addr);
		        // adjust the data length to now include the full packet length
		        pMsg->length = pMsg->length + MSG_HEADER_SIZE + MSG_FOOTER_SIZE;
		        // keep the DSN increasing 
		        pMsg->dsn = ++currentDSN;
		        // reset the time field
		        pMsg->time = 0;
		        // FCS bytes generated by CC2420
		        txlength = pMsg->length - MSG_FOOTER_SIZE;
		        txbufptr = pMsg;

				// put the packet data to the TXFIFO of the CC2420 immediately
		        if (post startSend()) {
		        	atomic {
						stateRadio = PRE_TX_STATE;
		        	}
		        	return SUCCESS;		        	
		        } 
	        }
	        return FAIL;
        }

        /**
         * Delayed RXFIFO is used to read the receive FIFO of the CC2420
         * in task context after the uC receives an interrupt that a packet
         * is in the RXFIFO.	Task context is necessary since reading from
         * the FIFO may take a while and we'd like to get other interrupts
         * during that time, or notifications of additional packets received
         * and stored in the CC2420 RXFIFO.
         */
        void delayedRXFIFO();

        task void delayedRXFIFOtask() {
	        delayedRXFIFO();
        }

        void delayedRXFIFO() {
	        uint8_t len = MSG_DATA_SIZE;
	        uint8_t _bPacketReceiving;
			
	        if ((!TOSH_READ_CC_FIFO_PIN()) && (!TOSH_READ_CC_FIFOP_PIN())) {
		        flushRXFIFO();
		        return;
	        }

	        atomic {
	                _bPacketReceiving = bPacketReceiving;

	                if (_bPacketReceiving) {
	                	if (!post delayedRXFIFOtask())
			               	flushRXFIFO();
	                } else {
		                bPacketReceiving = TRUE;
	                }
	        }

	        // JP NOTE: TODO: move readRXFIFO out of atomic context to permit
	        // high frequency sampling applications and remove delays on
	        // interrupts being processed.  There is a race condition
	        // that has not yet been diagnosed when RXFIFO may be interrupted.
	        if (!_bPacketReceiving) {
		        if (!call HPLChipconFIFO.readRXFIFO(len,(uint8_t*)rxbufptr)) {
			        atomic bPacketReceiving = FALSE;
			        if (!post delayedRXFIFOtask()) {
				        flushRXFIFO();
			        }
			        return;
		        }
	        }
	        flushRXFIFO();
        }

        /**********************************************************
         * FIFOP lo Interrupt: Rx data avail in CC2420 fifo
         * Radio must have been in Rx mode to get this interrupt
         * If FIFO pin =lo then fifo overflow=> flush fifo & exit
         * 
         *
         * Things ToDo:
         *
         * -Disable FIFOP interrupt until PacketRcvd task complete 
         * until send.done complete
         *
         * -Fix mixup: on return
         *  rxbufptr->rssi is CRC + Correlation value
         *  rxbufptr->strength is RSSI
         **********************************************************/
        async event result_t FIFOP.fired() {

	        /** Check for RXFIFO overflow **/
	        if (!TOSH_READ_CC_FIFO_PIN()){
		        flushRXFIFO();
		        return SUCCESS;
	        }

	        atomic {
	                if (post delayedRXFIFOtask()) {
	                call FIFOP.disable();
	                }
	                else {
		                flushRXFIFO();
	                }
	        }

	        // return SUCCESS to keep FIFOP events occurring
	        return SUCCESS;
        }

        /**
         * After the buffer is received from the RXFIFO,
         * process it, then post a task to signal it to the higher layers
         */
        async event result_t HPLChipconFIFO.RXFIFODone(uint8_t length, uint8_t *data) {
	        // JP NOTE: rare known bug in high contention:
	        // radio stack will receive a valid packet, but for some reason the
	        // length field will be longer than normal.	 The packet data will
	        // be valid up to the correct length, and then will contain garbage
	        // after the correct length.  There is no currently known fix.
	        uint8_t currentstate;
	        atomic {
	        	currentstate = stateRadio;
	        }
			
	        // if a FIFO overflow occurs or if the data length is invalid, flush
	        // the RXFIFO to get back to a normal state.
	        if ((!TOSH_READ_CC_FIFO_PIN() && !TOSH_READ_CC_FIFOP_PIN())
	                        || (length == 0) || (length > MSG_DATA_SIZE)) {
		        flushRXFIFO();
		        atomic bPacketReceiving = FALSE;
		        return SUCCESS;
	        }

	        rxbufptr = (TOS_MsgPtr)data;

	        // if the length is shorter, we have to move the CRC bytes
	        rxbufptr->crc = data[length-1] >> 7;

	        // check for invalid packets
	        // an invalid packet is a non-data packet with the wrong
	        // addressing mode (FCFLO byte)
	        // NOTE: Now, do not check crc here
	        if (((rxbufptr->fcfhi & 0x07) != CC2420_DEF_FCF_TYPE_DATA) ||
	                        (rxbufptr->fcflo != CC2420_DEF_FCF_LO) /*||
							rxbufptr->crc != 1*/) {
		        flushRXFIFO();
		        atomic bPacketReceiving = FALSE;
		        return SUCCESS;
	        }

	        rxbufptr->length = rxbufptr->length - MSG_HEADER_SIZE - MSG_FOOTER_SIZE;

	        if (rxbufptr->length > TOSH_DATA_LENGTH) {
		        flushRXFIFO();
		        atomic bPacketReceiving = FALSE;
		        return SUCCESS;
	        }

	        // adjust destination to the right byte order
	        rxbufptr->addr = fromLSB16(rxbufptr->addr);

	        // put in RSSI
	        rxbufptr->strength = data[length-2];
	        // put in LQI
	        rxbufptr->lqi = data[length-1] & 0x7F;

	        atomic {
	            if (!post PacketRcvd()) {
	                bPacketReceiving = FALSE;
                }
        	}

        	if ((!TOSH_READ_CC_FIFO_PIN()) && (!TOSH_READ_CC_FIFOP_PIN())) {
		        flushRXFIFO();
		        return SUCCESS;
	        }

	        if (!(TOSH_READ_CC_FIFOP_PIN())) {
		        if (post delayedRXFIFOtask())
			        return SUCCESS;
	        }
	        flushRXFIFO();
			
	        return SUCCESS;
        }

        /**
         * Notification that the TXFIFO has been filled with the data from the packet
         * Next step is to try to send the packet
         */
        async event result_t HPLChipconFIFO.TXFIFODone(uint8_t length, uint8_t *data) {
			sendPacket();
	        return SUCCESS;
        }

        // Default events for radio send/receive coordinators do nothing.
        // Be very careful using these, you'll break the stack.
        // The "byte()" event is never signalled because the CC2420 is a packet
        // based radio.
        default async event void RadioSendCoordinator.startSymbol(uint8_t bitsPerBlock, uint8_t offset, TOS_MsgPtr msgBuff) { }
        default async event void RadioSendCoordinator.byte(TOS_MsgPtr msg, uint8_t byteCount) { }
        default async event void RadioReceiveCoordinator.startSymbol(uint8_t bitsPerBlock, uint8_t offset, TOS_MsgPtr msgBuff) { }
        default async event void RadioReceiveCoordinator.byte(TOS_MsgPtr msg, uint8_t byteCount) { }


        /**********************************************************
         * Carrier Sense interface implementation 
         * Enable reading of CCA every clock tick
         * 		  
		 */
		async command result_t WiDOMCarrierSense.start(/*uint16_t clkTks*/) {
			bool _CCA;

			// the CCA pin will only be valid after rx calibrate 
			// this takes takes about 12 symbols (128us).
			// To know when this has passed, will be polling the state 
			// of the radio until the RSSI is valid
			atomic _CCA = CCA; 

			if (_CCA == FALSE) { // if CCA = TRUE, then carrier sensing is activated already
				call HPLChipcon.cmd(CC2420_SRXON);	
				atomic {
					CCA = TRUE;
					CCAValidTime = 6; // takes 192us to switch to RX mode and have a valid CCA
					CCAState = INVALID_CCA_STATE; // this will always make either the idle or busy event to be signalled
//					CCAState = 1; // assume channel is idle   
					taskCheckCCA = FALSE;
				}	
				return call ClockTicks.postTicks(TRUE);
			}
			return SUCCESS;
		}

		async command result_t WiDOMCarrierSense.stop() {
			call HPLChipcon.cmd(CC2420_SRFOFF); // set radio to idle?	
			atomic {
				CCA = FALSE;
			}
			return call ClockTicks.postTicks(FALSE);
		}

        /**********************************************************
         * Signal edges of CCA pin 
         * 
         */		 		  
		task void CheckCCA() {
			uint8_t _CCAState, newCCAState=1;
			
			newCCAState = TOSH_READ_CC_CCA_PIN();
			atomic _CCAState = CCAState;
			if (_CCAState != newCCAState) { 
				if (newCCAState == 0) signal WiDOMCarrierSense.channelBusy();
				else signal WiDOMCarrierSense.channelIdle();
			}
			atomic {
				CCAState = newCCAState; 
				taskCheckCCA = FALSE;
			}			
		}

        /**********************************************************
         * WiDOM clock ticks handler 
         * Only enabled during carrier sensing time interval
         * 
		 */
		async event void ClockTicks.tick() {
			atomic {
				if (CCAValidTime) CCAValidTime--;
				else if (CCA == TRUE && taskCheckCCA == FALSE) {
					taskCheckCCA = TRUE;
					post CheckCCA();
				}
			}
		}

        /**********************************************************
		 * set the radio into tx test mode, to send a modulated carrier
		 * call this before sending carrier pulses
		 */		 
		async command result_t WiDOMCarrierPulse.carrierTXModeStart() {
			uint8_t currentstate;
	        atomic currentstate = stateRadio;
						
	        if (currentstate != CARRIER_TX_STATE) {
	        	call HPLChipcon.cmd(CC2420_SRFOFF); //stop radio
				call HPLChipcon.write(CC2420_MDMCTRL1, 0x050C); // MDMCTRL1 with TX_MODE=3 : Carrier Test Mode
				call HPLChipcon.write(CC2420_DACTST, 0x1800); // send unmodulated carrier
				flushRXFIFO();
				atomic {
					stateRadio = CARRIER_TX_STATE;
				}
				return SUCCESS;
			}		
			return FAIL;	
		}
	
        /**********************************************************
		 * set the radio into "normal" tx mode (buffered TXFIFO) 
		 * should also call this to get out of "carrier sensing" state, to be able to send		 
		 */		 
		async command result_t WiDOMCarrierPulse.carrierTXModeEnd() {
			uint8_t currentstate;
	        atomic currentstate = stateRadio;
						
	        if (currentstate == CARRIER_TX_STATE) {
	        	call WiDOMCarrierSense.stop();
	        	call HPLChipcon.cmd(CC2420_SRFOFF); //stop radio
    			call HPLChipcon.write(CC2420_MDMCTRL1, 0x0500); // default MDMCTRL1 value; TX_MODE=0 : buffered TXFIFO Mode
				call HPLChipcon.write(CC2420_DACTST, 0); // default value
				atomic stateRadio = IDLE_STATE;
				flushRXFIFO();
				call HPLChipcon.cmd(CC2420_SRXON); // by default, go to receive mode
				return SUCCESS;
			}		
			return FAIL;	
		}
		
        /**********************************************************
	     * start sending a carrier pulse
		 */		 
	   async command result_t WiDOMCarrierPulse.on() {
			return call HPLChipcon.cmd(CC2420_STXON); // tell radio to start sending
	   }
	
        /**********************************************************
	     * stop sending a carrier pulse
		 */		 
	   async command result_t WiDOMCarrierPulse.off() {
	        return call HPLChipcon.cmd(CC2420_SRFOFF); // stop radio
	   }
}
