// $Id: CC2420ControlM.nc,v 1.18 2005/11/30 23:30:41 jpolastre Exp $

/*									tab:4
 * "Copyright (c) 2000-2003 The Regents of the University  of California.  
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the author appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Copyright (c) 2002-2003 Intel Corporation
 * All rights reserved.
 *
 * This file is distributed under the terms in the attached INTEL-LICENSE     
 * file. If you do not find these files, copies can be found by writing to
 * Intel Research Berkeley, 2150 Shattuck Avenue, Suite 1300, Berkeley, CA, 
 * 94704.  Attention:  Intel License Inquiry.
 */
/*
 *
 * Authors:		Alan Broad, Joe Polastre
 * Date last modified:  $Revision: 1.18 $
 *
 * This module provides the CONTROL functionality for the 
 * Chipcon2420 series radio. It exports both a standard control 
 * interface and a custom interface to control CC2420 operation.
 */

/**
 * @author Alan Broad, Crossbow
 * @author Joe Polastre
 */
 
#define _registerDftVal( _val_ )   ( CC2420_DFTVAL_##_val_ )
#define _dftVal( _bit_pos_ )   ( CC2420_DFTVAL_##_bit_pos_ << CC2420_##_bit_pos_ )


includes CC2420Const;
includes byteorder;


module CC2420ControlM {
  provides {
    interface SplitControl;
    interface CC2420Control;
  }
  uses {
    interface StdControl as HPLChipconControl;
    interface HPLCC2420 as HPLChipcon;
    interface HPLCC2420RAM as HPLChipconRAM;
    interface HPLCC2420Interrupt as CCA;
  }
}
implementation
{

  enum {
    IDLE_STATE = 0,
    INIT_STATE,
    INIT_STATE_DONE,
    START_STATE,
    START_STATE_DONE,
    STOP_STATE,
  };

  uint8_t state = 0;
  norace uint16_t gCurrentParameters[CC2420_CP_NUM_POS];

   /************************************************************************
   * SetRegs
   *  - Configure CC2420 registers with current values
   *  - Readback 1st register written to make sure electrical connection OK
   *************************************************************************/
  bool SetRegs(){
    uint16_t data;
	      
    call HPLChipcon.write(CC2420_MAIN,gCurrentParameters[CP_MAIN]);   		    
    call HPLChipcon.write(CC2420_MDMCTRL0, gCurrentParameters[CP_MDMCTRL0]);
    data = call HPLChipcon.read(CC2420_MDMCTRL0);
    if (data != gCurrentParameters[CP_MDMCTRL0]) return FALSE;
    
    call HPLChipcon.write(CC2420_MDMCTRL1, gCurrentParameters[CP_MDMCTRL1]);
    call HPLChipcon.write(CC2420_RSSI, gCurrentParameters[CP_RSSI]);
    call HPLChipcon.write(CC2420_SYNCWORD, gCurrentParameters[CP_SYNCWORD]);
    call HPLChipcon.write(CC2420_TXCTRL, gCurrentParameters[CP_TXCTRL]);
    call HPLChipcon.write(CC2420_RXCTRL0, gCurrentParameters[CP_RXCTRL0]);
    call HPLChipcon.write(CC2420_RXCTRL1, gCurrentParameters[CP_RXCTRL1]);
    call HPLChipcon.write(CC2420_FSCTRL, gCurrentParameters[CP_FSCTRL]);

    call HPLChipcon.write(CC2420_SECCTRL0, gCurrentParameters[CP_SECCTRL0]);
    call HPLChipcon.write(CC2420_SECCTRL1, gCurrentParameters[CP_SECCTRL1]);
    call HPLChipcon.write(CC2420_IOCFG0, gCurrentParameters[CP_IOCFG0]);
    call HPLChipcon.write(CC2420_IOCFG1, gCurrentParameters[CP_IOCFG1]);
    
    call HPLChipcon.write(CC2420_FSMTC, gCurrentParameters[CP_FSMTC]);
    call HPLChipcon.write(CC2420_MANAND, gCurrentParameters[CP_MANAND]);
    call HPLChipcon.write(CC2420_MANOR, gCurrentParameters[CP_MANOR]);
    call HPLChipcon.write(CC2420_AGCCTRL, gCurrentParameters[CP_AGCCTRL]);
    call HPLChipcon.write(CC2420_AGCTST0, gCurrentParameters[CP_AGCTST0]);
    call HPLChipcon.write(CC2420_AGCTST1, gCurrentParameters[CP_AGCTST1]);
    call HPLChipcon.write(CC2420_AGCTST2, gCurrentParameters[CP_AGCTST2]);
    call HPLChipcon.write(CC2420_FSTST0, gCurrentParameters[CP_FSTST0]);
    call HPLChipcon.write(CC2420_FSTST1, gCurrentParameters[CP_FSTST1]);
    call HPLChipcon.write(CC2420_FSTST2, gCurrentParameters[CP_FSTST2]);
    call HPLChipcon.write(CC2420_FSTST3, gCurrentParameters[CP_FSTST3]);
    call HPLChipcon.write(CC2420_RXBPFTST, gCurrentParameters[CP_RXBPFTST]);
    call HPLChipcon.write(CC2420_ADCTST, gCurrentParameters[CP_ADCTST]);
    call HPLChipcon.write(CC2420_DACTST, gCurrentParameters[CP_DACTST]);
    call HPLChipcon.write(CC2420_TOPTST, gCurrentParameters[CP_TOPTST]);	

    call HPLChipcon.cmd(CC2420_SFLUSHTX);    //flush Tx fifo
    call HPLChipcon.cmd(CC2420_SFLUSHRX);
 
    return TRUE;
  }

  task void taskInitDone() {
    signal SplitControl.initDone();
  }

  task void taskStopDone() {
    signal SplitControl.stopDone();
  }

  task void PostOscillatorOn() {
    //set freq, load regs
    SetRegs();
    call CC2420Control.setShortAddress(TOS_LOCAL_ADDRESS);
    call CC2420Control.TuneManual(((gCurrentParameters[CP_FSCTRL] << CC2420_FSCTRL_FREQ) & 0x1FF) + 2048);
    atomic state = START_STATE_DONE;
    signal SplitControl.startDone();
  }

  /*************************************************************************
   * Init CC2420 radio:
   *
   *************************************************************************/
  command result_t SplitControl.init() {

    uint8_t _state = FALSE;

    atomic {
      if (state == IDLE_STATE) {
	state = INIT_STATE;
	_state = TRUE;
      }
    }
    if (!_state)
      return FAIL;

    call HPLChipconControl.init();
  
	// Set default parameters
    gCurrentParameters[CP_MAIN] = _registerDftVal(CPMAIN);
    gCurrentParameters[CP_MDMCTRL0] = 
		_dftVal(MDMCTRL0_FRAME) |
		_dftVal(MDMCTRL0_PANCRD) | 
		_dftVal(MDMCTRL0_ADRDECODE) |
		_dftVal(MDMCTRL0_CCAHIST) |
		_dftVal(MDMCTRL0_CCAMODE) |
		_dftVal(MDMCTRL0_AUTOCRC) |
		_dftVal(MDMCTRL0_AUTOACK) |
		_dftVal(MDMCTRL0_PREAMBL);

    gCurrentParameters[CP_MDMCTRL1] = 
		_dftVal(MDMCTRL1_CORRTHRESH) |
		_dftVal(MDMCTRL1_DEMOD_MODE) |
		_dftVal(MDMCTRL1_MODU_MODE) |
		_dftVal(MDMCTRL1_TX_MODE) |
		_dftVal(MDMCTRL1_RX_MODE);
		
    gCurrentParameters[CP_RSSI] = _registerDftVal(RSSI);
    gCurrentParameters[CP_SYNCWORD] = _registerDftVal(RSSI_SYNCWORD);

    gCurrentParameters[CP_TXCTRL] = 
		_dftVal(TXCTRL_BUFCUR) |
		_dftVal(TXCTRL_TURNARND) |
		_dftVal(TXCTRL_VAR) |
		_dftVal(TXCTRL_XMITCUR) |
		_dftVal(TXCTRL_PACUR) |
		_dftVal(TXCTRL_PADIFF) |
		(CC2420_DEF_RFPOWER << CC2420_TXCTRL_PAPWR);

    gCurrentParameters[CP_RXCTRL0] = 
		_dftVal(RXCTRL0_BUFCUR) | 
		_dftVal(RXCTRL0_HILNAG) |
		_dftVal(RXCTRL0_MLNAG) | 
		_dftVal(RXCTRL0_LOLNAG) | 
		_dftVal(RXCTRL0_HICUR) | 
		_dftVal(RXCTRL0_MCUR) | 
		_dftVal(RXCTRL0_LOCUR);

    gCurrentParameters[CP_RXCTRL1]  = 
		_dftVal(RXCTRL1_LOCUR) |
		_dftVal(RXCTRL1_MIDCUR) |
		_dftVal(RXCTRL1_LOLOGAIN) |
		_dftVal(RXCTRL1_MEDLOGAIN) |
		_dftVal(RXCTRL1_HIHGM) |
		_dftVal(RXCTRL1_MEDHGM) |
		_dftVal(RXCTRL1_LNACAP) |
		_dftVal(RXCTRL1_RMIXT) |
		_dftVal(RXCTRL1_RMIXV) |
		_dftVal(RXCTRL1_RMIXCUR);

    gCurrentParameters[CP_FSCTRL]   =
		_dftVal(FSCTRL_LOCK) |
		_dftVal(FSCTRL_CALDONE) |
		_dftVal(FSCTRL_CALRUNING) |
		_dftVal(FSCTRL_LOCKLEN) |
		_dftVal(FSCTRL_LOCKSTAT) |
		((357+5*(CC2420_DEF_CHANNEL-11)) << CC2420_FSCTRL_FREQ);

    gCurrentParameters[CP_SECCTRL0] = 
		_dftVal(SECCTRL0_PROTECT) |
		_dftVal(SECCTRL0_CBCHEAD) |
		_dftVal(SECCTRL0_SAKEYSEL) |
		_dftVal(SECCTRL0_TXKEYSEL) |
		_dftVal(SECCTRL0_RXKEYSEL) |
		_dftVal(SECCTRL0_SECM) |
		_dftVal(SECCTRL0_SECMODE);

    gCurrentParameters[CP_SECCTRL1] = _registerDftVal(SECCTRL1);
    gCurrentParameters[CP_BATTMON]  = _registerDftVal(BATTMON);

    // set fifop threshold to greater than size of tos msg, 
    // fifop goes active at end of msg
    gCurrentParameters[CP_IOCFG0]   = 
		_dftVal(IOCFG0_FIFOPOL) |
		_dftVal(IOCFG0_FIFOPPOL) |
		_dftVal(IOCFG0_SFD) |
		_dftVal(IOCFG0_CCAPOL) |
		_dftVal(IOCFG0_FIFOTHR);

    gCurrentParameters[CP_IOCFG1]   =  _registerDftVal(IOCFG1);

    gCurrentParameters[CP_FSMTC] =  _registerDftVal(FSMTC);
    gCurrentParameters[CP_MANAND] =  _registerDftVal(MANAND);
    gCurrentParameters[CP_MANOR] =  _registerDftVal(MANOR);
    gCurrentParameters[CP_AGCCTRL] =  _registerDftVal(AGCCTRL);
    gCurrentParameters[CP_AGCTST0] =  _registerDftVal(AGCTST0);
    gCurrentParameters[CP_AGCTST1] =  _registerDftVal(AGCTST1);
    gCurrentParameters[CP_AGCTST2] =  _registerDftVal(AGCTST2);
    gCurrentParameters[CP_FSTST0] =  _registerDftVal(FSTST0);
    gCurrentParameters[CP_FSTST1] =  _registerDftVal(FSTST1);
    gCurrentParameters[CP_FSTST2] =  _registerDftVal(FSTST2);
    gCurrentParameters[CP_FSTST3] =  _registerDftVal(FSTST3);
    gCurrentParameters[CP_RXBPFTST] =  _registerDftVal(RXBPFTST);
    gCurrentParameters[CP_ADCTST] =  _registerDftVal(ADCTST);
    gCurrentParameters[CP_DACTST] =  _registerDftVal(DACTST);
    gCurrentParameters[CP_TOPTST] =  _registerDftVal(TOPTST);
    
    atomic state = INIT_STATE_DONE;
    post taskInitDone();
    return SUCCESS;
  }

  command result_t SplitControl.stop() {
    result_t ok;
    uint8_t _state = FALSE;

    atomic {
      if (state == START_STATE_DONE) {
		state = STOP_STATE;
		_state = TRUE;
      }
    }
    if (!_state)
      return FAIL;

    call HPLChipcon.cmd(CC2420_SXOSCOFF); 
    ok = call CCA.disable();
    ok &= call HPLChipconControl.stop();

    TOSH_CLR_CC_RSTN_PIN();
    ok &= call CC2420Control.VREFOff();
    TOSH_SET_CC_RSTN_PIN();

    if (ok)
      post taskStopDone();
    
    atomic state = INIT_STATE_DONE;
    return ok;
  }

/******************************************************************************
 * Start CC2420 radio:
 * -Turn on 1.8V voltage regulator, wait for power-up, 0.6msec
 * -Release reset line
 * -Enable CC2420 crystal,          wait for stabilization, 0.9 msec
 *
 ******************************************************************************/

  command result_t SplitControl.start() {
    result_t status;
    uint8_t _state = FALSE;

    atomic {
      if (state == INIT_STATE_DONE) {
		state = START_STATE;
		_state = TRUE;
      }
    }
    if (!_state)
      return FAIL;

    call HPLChipconControl.start();
    //turn on power
    call CC2420Control.VREFOn();
    // toggle reset
    TOSH_CLR_CC_RSTN_PIN();
    TOSH_wait();
    TOSH_SET_CC_RSTN_PIN();
    TOSH_wait();
    // turn on crystal, takes about 860 usec, 
    // chk CC2420 status reg for stablize
    status = call CC2420Control.OscillatorOn();

    return status;
  }

  /*************************************************************************
   * TunePreset
   * -Set CC2420 channel
   * Valid channel values are 11 through 26.
   * The channels are calculated by:
   *  Freq = 2405 + 5(k-11) MHz for k = 11,12,...,26
   * chnl requested 802.15.4 channel 
   * return Status of the tune operation
   *************************************************************************/
  command result_t CC2420Control.TunePreset(uint8_t chnl) {
    int fsctrl;
    uint8_t status;
    
    fsctrl = 357 + 5*(chnl-11);
    gCurrentParameters[CP_FSCTRL] = (gCurrentParameters[CP_FSCTRL] & 0xfc00) | (fsctrl << CC2420_FSCTRL_FREQ);
    status = call HPLChipcon.write(CC2420_FSCTRL, gCurrentParameters[CP_FSCTRL]);
    // if the oscillator is started, recalibrate for the new frequency
    // if the oscillator is NOT on, we should not transition to RX mode
    if (status & (1 << CC2420_XOSC16M_STABLE))
      call HPLChipcon.cmd(CC2420_SRXON);
    return SUCCESS;
  }

  /*************************************************************************
   * TuneManual
   * Tune the radio to a given frequency. Frequencies may be set in
   * 1 MHz steps between 2400 MHz and 2483 MHz
   * 
   * Desiredfreq The desired frequency, in MHz.
   * Return Status of the tune operation
   *************************************************************************/
  command result_t CC2420Control.TuneManual(uint16_t DesiredFreq) {
    int fsctrl;
    uint8_t status;
   
    fsctrl = DesiredFreq - 2048;
    gCurrentParameters[CP_FSCTRL] = (gCurrentParameters[CP_FSCTRL] & 0xfc00) | (fsctrl << CC2420_FSCTRL_FREQ);
    status = call HPLChipcon.write(CC2420_FSCTRL, gCurrentParameters[CP_FSCTRL]);
    // if the oscillator is started, recalibrate for the new frequency
    // if the oscillator is NOT on, we should not transition to RX mode
    if (status & (1 << CC2420_XOSC16M_STABLE))
      call HPLChipcon.cmd(CC2420_SRXON);
    return SUCCESS;
  }

  /*************************************************************************
   * Get the current frequency of the radio
   */
  command uint16_t CC2420Control.GetFrequency() {
    return ((gCurrentParameters[CP_FSCTRL] & (0x1FF << CC2420_FSCTRL_FREQ))+2048);
  }

  /*************************************************************************
   * Get the current channel of the radio
   */
  command uint8_t CC2420Control.GetPreset() {
    uint16_t _freq = (gCurrentParameters[CP_FSCTRL] & (0x1FF << CC2420_FSCTRL_FREQ));
    _freq = (_freq - 357)/5;
    _freq = _freq + 11;
    return _freq;
  }

  /*************************************************************************
   * TestTxMode
   * Set the CC2420 Radio into transmit test mode (does not start transmit).
   * return SUCCESS if the radio was successfully switched to TX mode.
   *************************************************************************/
/*   
  async command result_t CC2420Control.TestTxMode() {
	result_t res = FAIL;
	res = call HPLChipcon.write(CC2420_MDMCTRL1, 0x0508);
	res = call HPLChipcon.write(CC2420_DACTST, 0x1800);
	return res;
  }
*/
  /*************************************************************************
   * NormalTxMode
   * Set the CC2420 Radio into "normal "transmit mode (does not start transmit).
   * return SUCCESS if the radio was successfully switched to TX mode.
   *************************************************************************/
/*   
  async command result_t CC2420Control.NormalTxMode() {
	result_t res = FAIL;
	// assumes values in gCurrentParameters are set for "normal" tx mode
    res = call HPLChipcon.write(CC2420_MDMCTRL1, gCurrentParameters[CP_MDMCTRL1]);
	res = call HPLChipcon.write(CC2420_DACTST, 0);
	return res;
  }
*/
  /*************************************************************************
   * TxMode (STXON)
   * Shift the CC2420 Radio into transmit mode (start transmit).
   * return SUCCESS if the radio was successfully switched to TX mode.
   *************************************************************************/
  async command result_t CC2420Control.TxMode() {
    call HPLChipcon.cmd(CC2420_STXON);
    return SUCCESS;
  }

  /*************************************************************************
   * TxModeOnCCA (STXONCCA)
   * Shift the CC2420 Radio into transmit mode when the next clear channel
   * is detected.
   *
   * return SUCCESS if the transmit request has been accepted
   *************************************************************************/
  async command result_t CC2420Control.TxModeOnCCA() {
   call HPLChipcon.cmd(CC2420_STXONCCA);
   return SUCCESS;
  }

  /*************************************************************************
   * RxMode
   * Shift the CC2420 Radio into receive mode 
   *************************************************************************/
  async command result_t CC2420Control.RxMode() {
    call HPLChipcon.cmd(CC2420_SRXON);
    return SUCCESS;
  }

  /*************************************************************************
   * SetRFPower
   * power = 31 => full power    (0dbm)
   *          3 => lowest power  (-25dbm)
   * return SUCCESS if the radio power was successfully set
   *************************************************************************/
  command result_t CC2420Control.SetRFPower(uint8_t power) {
    gCurrentParameters[CP_TXCTRL] = (gCurrentParameters[CP_TXCTRL] & (~CC2420_TXCTRL_PAPWR_MASK)) | (power << CC2420_TXCTRL_PAPWR);
    call HPLChipcon.write(CC2420_TXCTRL,gCurrentParameters[CP_TXCTRL]);
    return SUCCESS;
  }

  /*************************************************************************
   * GetRFPower
   * return power seeting
   *************************************************************************/
  command uint8_t CC2420Control.GetRFPower() {
    return (gCurrentParameters[CP_TXCTRL] & CC2420_TXCTRL_PAPWR_MASK); //rfpower;
  }

  async command result_t CC2420Control.OscillatorOn() {
    uint16_t i;
    uint8_t status;

    i = 0;

    // uncomment to measure the startup time from 
    // high to low to high transitions
    // output "1" on the CCA pin
#ifdef CC2420_MEASURE_OSCILLATOR_STARTUP
      call HPLChipcon.write(CC2420_IOCFG1, 31);
      // output oscillator stable on CCA pin
      // error in CC2420 datasheet 1.2: SFDMUX and CCAMUX incorrectly labelled
      TOSH_uwait(50);
#endif

    call HPLChipcon.write(CC2420_IOCFG1, 24);

    // have an event/interrupt triggered when it starts up
    call CCA.startWait(TRUE);
    
    // start the oscillator
    status = call HPLChipcon.cmd(CC2420_SXOSCON);   //turn-on crystal

    return SUCCESS;
  }

  async command result_t CC2420Control.OscillatorOff() {
    call HPLChipcon.cmd(CC2420_SXOSCOFF);   //turn-off crystal
    return SUCCESS;
  }

  async command result_t CC2420Control.VREFOn(){
    TOSH_SET_CC_VREN_PIN();                    //turn-on  
    // TODO: JP: measure the actual time for VREF to stabilize
    TOSH_uwait(600);  // CC2420 spec: 600us max turn on time
    return SUCCESS;
  }

  async command result_t CC2420Control.VREFOff(){
    TOSH_CLR_CC_VREN_PIN();                    //turn-off  
    return SUCCESS;
  }

  async command result_t CC2420Control.enableAutoAck() {
    gCurrentParameters[CP_MDMCTRL0] |= (1 << CC2420_MDMCTRL0_AUTOACK);
    return call HPLChipcon.write(CC2420_MDMCTRL0,gCurrentParameters[CP_MDMCTRL0]);
  }

  async command result_t CC2420Control.disableAutoAck() {
    gCurrentParameters[CP_MDMCTRL0] &= ~(1 << CC2420_MDMCTRL0_AUTOACK);
    return call HPLChipcon.write(CC2420_MDMCTRL0,gCurrentParameters[CP_MDMCTRL0]);
  }

  async command result_t CC2420Control.enableAddrDecode() {
    gCurrentParameters[CP_MDMCTRL0] |= (1 << CC2420_MDMCTRL0_ADRDECODE);
    return call HPLChipcon.write(CC2420_MDMCTRL0,gCurrentParameters[CP_MDMCTRL0]);
  }

  async command result_t CC2420Control.disableAddrDecode() {
    gCurrentParameters[CP_MDMCTRL0] &= ~(1 << CC2420_MDMCTRL0_ADRDECODE);
    return call HPLChipcon.write(CC2420_MDMCTRL0,gCurrentParameters[CP_MDMCTRL0]);
  }

  command result_t CC2420Control.setShortAddress(uint16_t addr) {
    addr = toLSB16(addr);
    return call HPLChipconRAM.write(CC2420_RAM_SHORTADR, 2, (uint8_t*)&addr);
  }

  async event result_t HPLChipconRAM.readDone(uint16_t addr, uint8_t length, uint8_t* buffer) {
     return SUCCESS;
  }

  async event result_t HPLChipconRAM.writeDone(uint16_t addr, uint8_t length, uint8_t* buffer) {
     return SUCCESS;
  }

   async event result_t CCA.fired() {
     // reset the CCA pin back to the CCA function
     call HPLChipcon.write(CC2420_IOCFG1, 0);
     post PostOscillatorOn();
     return FAIL;
   }
	
}

