// CalcNNodes.cpp 
// Simulate number of nodes estimation
//
// @author Nuno Pereira
//

#include <iostream>
#include <tchar.h>
#include <math.h>
#include <time.h>
#include <stdlib.h>
#include "cRandomLFSR.h"
#include "cMT19937.h"

// Number of random numbers to generate
#define K_DEFAULT 5
int K=K_DEFAULT;

// Number of estimations to do for each simulation
#define NEST_MAX 1000
int NEST=NEST_MAX;

//#define RSEED ((unsigned)time( NULL ))
#define RSEED ((unsigned)12345)

//#define NODE_RND_SEED (i+1)
#define NODE_RND_SEED (NNODES+i+1)
//#define NODE_RND_SEED (rand()+i+1)

// Max number of nodes (m)
const int NNODES_MAX=pow(2,16);
//const int NNODES_MAX=16384;
//const int NNODES_MAX=1024;

// we will do experiments from 1 to NNODES_MAX
// growing in powers of 2
const int NSIM = (int)((log10(((double)NNODES_MAX)) / log10(2.0))) + 1;

// number of nodes in each simulation
// this will be 1, 2, 4, 8, 16, 32, 64, ..., NNODES_MAX
int NNODES;

// what kind of PRNG to use
//#define PRNG_16_BIT		// 16 bit Random LFSR PRNG, implementation like TinyOS
#define PRNG_32_BIT			// 32 bit Mersenne Twister PRNG, implementation by Makoto Matsumoto and Takuji Nishimura

#ifdef PRNG_32_BIT
	typedef unsigned long rnType;
	typedef cMT19937 prngType;
	 
	// Maximum random value generated (this depends on the PRNG used)
	#define MAXV 0xFFFFFFFF
#endif

#ifdef PRNG_16_BIT
	typedef unsigned __int16 rnType;
	typedef cRandomLFSR prngType;

	// Maximum random value generated (this depends on the PRNG used)
	#define MAXV 0xFFFF
#endif

// colect statistics on the estimations
#define STATS_ESTIMATE

// colect statistics on the probability of estimations
//#define STATS_INTERVAL

// round a double
double round(double doValue, int nPrecision)
{
	static const double doBase = 10.0;
	double doComplete5, doComplete5i;
	
	doComplete5 = doValue * pow(doBase, (double) (nPrecision + 1));
	
	if(doValue < 0.0)
		doComplete5 -= 5.0;
	else
		doComplete5 += 5.0;
	
	doComplete5 /= doBase;
	modf(doComplete5, &doComplete5i);
	
	return doComplete5i / pow(doBase, (double) nPrecision);
}

// compare two integers (for qsort routine)
int compare( const void *arg1, const void *arg2 )
{
	return *((int*)arg1)-*((int*)arg2);
}

// compare two doubles (for qsort routine)
int compared( const void *arg1, const void *arg2 )
{
/*
	if (round(*((double*)arg1),5) > round(*((double*)arg2),5)) return 1;
	if (round(*((double*)arg1),5) < round(*((double*)arg2),5)) return -1;
*/
	if (*((double*)arg1) > *((double*)arg2)) return 1;
	if (*((double*)arg1) < *((double*)arg2)) return -1;
	return 0;
}

// return the value in the percent^th position from 
// a vector of integers (assumes that the quantity is 
// also an integer)
int percentile(int *v, int size, double percent)
{
	int pos = (int)(size * percent);

	if (((double)pos) == round(size * percent, 5))
		return v[pos-1];

	return v[pos];
}

// return the value in the percent^th position from 
// a vector of doubles
double percentiled(double *v, int size, double percent)
{
	int pos = (int)(size * percent);
	if (((double)pos) == round(size * percent, 5))
		return v[pos];

	return (v[pos] + v[pos+1]) / 2.0;
}

// This class main purpose is to hold a different
// random number generator for each node.
// If the random numbers in the nodes are independent
// depends on the call to method initialize (see NODE_RND_SEED)
class cNode 
{
protected:
	
	prngType random;				// PRNG used for the node

	int id;							// id of the node 

	rnType *r;						// array of k random numbers generated

	// static members
	static  rnType *x;				// array of k minimum random numbers generated amongst all nodes
	static cNode *nodes;			// hold reference to nodes array

public:
	cNode(void) {
		r = new rnType[K];
	};
	
	~cNode(void) { 
	}

	void initialize(cNode *_nodes, int nodeId) {
		id = nodeId;
		nodes = _nodes;
		random.initialize(id);
	}

	static int doEstimation() {
		if (x == NULL) x = new rnType[K];

		// generate random numbers
		for (int i = 0; i < NNODES; i++) {
			for (int q = 0; q < K; q++) {
				nodes[i].r[q] = (nodes[i].random.rand());
				//printf("\nnodes[%d].r[%d]=%u;", i, q, nodes[i].r[q]);
				//nodes[i].r[q] = rand();
			}
			//printf("\n");
		}

		// compute minimum random numbers generated
		for (int q = 0; q < K; q++) {
			x[q] = nodes[0].r[q];
			for (i = 1; i < NNODES; i++) {
				if (nodes[i].r[q] < x[q]) x[q] = nodes[i].r[q];
			}
		}
/*
		for (int q = 0; q < K; q++) {
			printf("\nx[%d]=%u;", q, x[q]);
		}
		printf("\n");

		exit(0);
*/
		return ML_estimation();
	}

	// implementation of fig(5)
	static double doProbabilityEstimation(int j1, int j2)
	{
		double Q,fj1,fj2,denom;
		double nnodes_double;
		Q = compute_Q();
		fj1 = compute_f(j1,Q);
		fj2 = compute_f(j2,Q);
		denom = compute_f(1,Q);
		nnodes_double = (fj1-fj2)/denom;
		return nnodes_double;
	}

private:
	static double probability(int j) {
		double prod = 1;
		
		for (int q=0; q < K; q++) {
			prod *= ( pow((1-(x[q]-1)/((double)MAXV)), j) - pow((1-(x[q])/((double)MAXV)), j) );
		}
		return prod;
	}

	// implementation of (17)
	static int ML_estimation() {
		double sumv=0;

		for (int q = 0; q < K; q++)
			sumv += log( 1 / ( 1 - (x[q]-1) / (double)MAXV ) );

		int h1 = (int) (K / sumv) ;
		int h2 = (int) h1 + 1;
		if ( probability(h1) > probability(h2) ) return h1;
		else return h2;
	}	

	static double compute_Q()
	{
		int q;
		double prod;
		prod = 1.0;
		for (q=0;q<K;q++) {
			prod = prod * (1-((double) x[q])/MAXV);    
		}
		return prod;
	}

	static double compute_f( int j, double Q)
	{
		int q;
		double sum,term;
		double invj;
		sum = 0;
		term = pow((double)j/MAXV,K)*pow(Q,j-1);
		sum = sum + term;
		for (q=K-1;q>=0;q--) {
			invj = 1.0/j; 
			term = term*(q+1)*invj*(1.0/log(Q))*(-1);
			sum  = sum + term;
		}
		return sum;
	}
};

// init static members
rnType *cNode::x = NULL;
cNode *cNode::nodes = NULL;		

int main(int argc , char *argv[ ])
{
	cNode (*nodes) = new cNode[NNODES_MAX];
	int *nnodes = new int[NSIM];

#ifdef STATS_ESTIMATE
	int (*est)[NEST_MAX] = new int[NSIM][NEST_MAX];
//	double *avg = new double[NSIM];
//	double *stdev = new double[NSIM];
	int *q1 = new int[NSIM];
	int *min = new int[NSIM];
	int *median = new int[NSIM];
	int *max = new int[NSIM];
	int *q3 = new int[NSIM];
#endif
#ifdef STATS_INTERVAL
	double (*prob_est)[NEST_MAX] = new double[NSIM][NEST_MAX];
	double *prob_q1 = new double[NSIM];
	double *prob_min = new double[NSIM];
	double *prob_median = new double[NSIM];
	double *prob_max = new double[NSIM];
	double *prob_q3 = new double[NSIM];
	int *nestinj1j2 = new int[NSIM]; // number of estimates within the interval [j1,j2]
#endif

	// check arguments
	if (argc > 1) {
		if (argc > 3) {
			printf ("\n Too Many Arguments\nUsage:\n\tcalnnodes k=<integer> nest=<integer> n=<integer>");
			printf("\n\tk = how many random numbers to generate\n\tnest or n = Number of estimations to do for each simulation\n");
			exit(1);
		}		
	}

	srand( RSEED );

	NNODES=1;
	for (int j=0; j<NSIM; j++) {
		nnodes[j] = NNODES;

		// init random number generators of all nodes
		for (int i = 0; i < NNODES; i++) {
			nodes[i].initialize(nodes, NODE_RND_SEED);
		}

		// do NEST estimates and gather statistics
		int esti = cNode::doEstimation();
#ifdef STATS_ESTIMATE
		min[j] = max[j] = esti;
//		unsigned long long sum=0, sqrsum=0; // BUG: size of sqrsum often is not sufficient ! (only damages stdev calculation...)
#endif
#ifdef STATS_INTERVAL
		prob_min[j] = 1; prob_max[j] = 0; nestinj1j2[j] = 0; 
		int npest=0;
#endif
		for (i=0; i < NEST; i++) {
#ifdef STATS_ESTIMATE
			est[j][i] = esti;
//			sum+=est[j][i];
//			sqrsum+=est[j][i]*est[j][i];
			if (min[j] > est[j][i]) min[j] = est[j][i];
			if (max[j] < est[j][i]) max[j] = est[j][i];
#endif
#ifdef STATS_INTERVAL
			// estimation of probability for an interval
			int j1=(int)(NNODES*.5);
			int j2=(int)(NNODES*1.5);
			if (esti >= j1 && esti <= j2) nestinj1j2[j]++;
			double prob_esti = cNode::doProbabilityEstimation(j1, j2);
//			if (prob_esti > 0 && prob_esti <= 1) {
				prob_est[j][npest++] = prob_esti;
				if (prob_min[j] > prob_esti) prob_min[j] = prob_esti;
				if (prob_max[j] < prob_esti) prob_max[j] = prob_esti;
//			}
#endif
			esti = cNode::doEstimation();
		}
#ifdef STATS_ESTIMATE
//		avg[j] = ((double)sum) / NEST;
//		stdev[j] = sqrt(((double)sqrsum) / ((double)NEST) - avg[j] * avg[j]);

		// order list of estimation values
		qsort(est[j], NEST, sizeof(int), compare);

		// get  median, q1 and q3
		median[j] = percentile(est[j], NEST, 0.5);
		q1[j] = percentile(est[j], NEST, 0.25);
		q3[j] = percentile(est[j], NEST, 0.75);
#endif
#ifdef STATS_INTERVAL
		// order list of probability estimation values
		qsort(prob_est[j], npest, sizeof(double), compared);

		// get median, q1 and q3
		prob_median[j] = percentiled(prob_est[j], npest, 0.5);
		prob_q1[j] = percentiled(prob_est[j], npest, 0.25);
		prob_q3[j] = percentiled(prob_est[j], npest, 0.75);
#endif
		NNODES=(int)pow(2,j+1);
		fprintf(stderr, "%d/%d\n",j+1, NSIM);
	}

	// output results		
	printf("\nstat.\t");
	for (j=0; j<NSIM; j++) {
		printf("m=%d\t", nnodes[j]);
	}
#ifdef STATS_ESTIMATE
/*
	std::cout << "\navg\t";
	for (j=0; j<NSIM; j++) {
		printf("%.1f\t", avg[j]);
	}
	std::cout << "\nstdev\t";
	for (j=0; j<NSIM; j++) {
		printf("%.1f\t", stdev[j]);
	}
*/
	printf("\nq1\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", q1[j]);
	}
	printf("\nmin\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", min[j]);
	}
	printf("\nmedian\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", median[j]);
	}
	printf("\nmax\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", max[j]);
	}
	printf("\nq3\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", q3[j]);
	}
	printf("\n\n");
#endif
#ifdef STATS_INTERVAL
	printf("\nPq1\t");
	for (j=0; j<NSIM; j++) {
		printf("%.4f\t", prob_q1[j]);
	}
	printf("\nPmin\t");
	for (j=0; j<NSIM; j++) {
		printf("%.4f\t", prob_min[j]);
	}
	printf("\nPmedian\t");
	for (j=0; j<NSIM; j++) {
		printf("%.4f\t", prob_median[j]);
	}
	printf("\nPmax\t");
	for (j=0; j<NSIM; j++) {
		printf("%.4f\t", prob_max[j]);
	}
	printf("\nPq3\t");
	for (j=0; j<NSIM; j++) {
		printf("%.4f\t", prob_q3[j]);
	}
	printf("\n\nPj1j2\t");
	for (j=0; j<NSIM; j++) {
		printf("%d\t", (int)(((double)nestinj1j2[j]/NEST)*100));
	}
#endif

	delete [] nodes;
	delete nnodes;
#ifdef STATS_ESTIMATE
	delete [] est;
//	delete avg;
//	delete stdev;
	delete q1;
	delete min;
	delete median;
	delete max;
	delete q3;
#endif
#ifdef STATS_INTERVAL
	delete [] prob_est;
	delete prob_q1;
	delete prob_min;
	delete prob_median;
	delete prob_max;
	delete prob_q3;
	delete nestinj1j2;
#endif
	return 0;
}