/* ========================================================================== */
/*                                                                            */
/*   simple_rtc.c                                                             */
/*   (c) 2007 Nuno Pereira                                                    */
/*                                                                            */
/*   Description                                                              */
/*          Simple 32 bit timer for atmega128                                 */
/*          Uses Timer/Counter3                                               */
/* ========================================================================== */
#include "simple_rtc.h"

uint16_t simple_rtc_high=0;

//-------------------------------------------------------------------------------------------------------
//	SIGNAL(SIG_OVERFLOW)
//
//	DESCRIPTION:
//		When an overflow interrupt for Timer3 (TCNT3 register)
//		in atmega128 occurs, the 'high' part of the timer is incremented.
//
//  ARGUMENTS:
//      SIG_OVERFLOW1
//          Overflow Interrupt Signal for Timer1.
//-------------------------------------------------------------------------------------------------------
SIGNAL(SIG_OVERFLOW3) 
{
  simple_rtc_high++;
}

//-------------------------------------------------------------------------------------------------------
//	void simple_rtc_init()
//
//	DESCRIPTION:
//		Initializes and starts a high speed timer for atmega 128 (Timer/Counter3).
//		Enables overflow interrupt.
//
//-------------------------------------------------------------------------------------------------------
void simple_rtc_init() 
{
	simple_rtc_high=0;
	TCCR3A = 0; // stop timer
	TCNT3 = 0; // reset timer value
	TIFR3&=~BM(OCF3A); // clear OCF flag
	TIMSK3|=BM(TOIE3); // enable overflow interrupt
	TCCR3B = TCLK_CPU_DIV1; // set timer scale
}


//-------------------------------------------------------------------------------------------------------
//	void simple_rtc_reset()
//
//	DESCRIPTION:
//		Resets the already running timer (makes it ZERO).
//		(atmega 128 Timer/Counter3)
//
//-------------------------------------------------------------------------------------------------------
inline void simple_rtc_reset()
{
  cli(); // disable interrupts
  simple_rtc_high=0;
  TCNT3 = (uint16_t)0; // reset timer value
  sei(); // enable interrupts
}

//-------------------------------------------------------------------------------------------------------
//	void simple_rtc_stop()
//
//	DESCRIPTION:
//		Stops the already started timer.
//		Usefull for "stop-start" situations along with simple_rtc_start.
//		(atmega 128 Timer/Counter3)
//
//-------------------------------------------------------------------------------------------------------
inline void simple_rtc_stop()
{
  TCCR3B = 0; // set timer scale
}


//-------------------------------------------------------------------------------------------------------
//	void simple_rtc_start()
//
//	DESCRIPTION:
//		Starts the already initialized timer (with simple_rtc_init).
//		Usefull to resume counting after a simple_rtc_stop.
//		(atmega 128 Timer/Counter3)
//
//-------------------------------------------------------------------------------------------------------
inline void simple_rtc_start()
{
  TCCR3B = TCLK_CPU_DIV1; // set timer scale
}


//-------------------------------------------------------------------------------------------------------
//	uint32_t simple_rtc_get()
//
//	DESCRIPTION:
//		Returns timer value stored in TCNT1 register.
//		(atmega 128 Timer/Counter3)
//
//-------------------------------------------------------------------------------------------------------
inline uint32_t simple_rtc_get() 
{
  uint32_t time;
  cli(); // disable interrupts
  time = ((uint32_t)((uint32_t)simple_rtc_high) << 16) + ((uint32_t)(TCNT3));
  sei(); // enable interrupts

  return time;
}

inline void simple_rtc_wait_until(uint32_t when_clkTks) 
{
 	uint32_t time;
	do {
		time = ((uint32_t)((uint32_t)simple_rtc_high) << 16) + ((uint32_t)(TCNT3));
		if (time >= when_clkTks) break;
	} while (1);
}

inline void simple_rtc_wait_for(uint32_t time_clkTks) 
{
 	uint32_t time = ((uint32_t)((uint32_t)simple_rtc_high) << 16) + ((uint32_t)(TCNT3));
	uint32_t when = time + time_clkTks;

	simple_rtc_wait_until(when);
}
