//******************************************************************************
//! CAN Interpolation
//! 
//! IPP-HURRAY! http://www.hurray.isep.ipp.pt
//! 
//!
//!
//! Contributing authors:
//!				Nuno Pereira
//!				Nuno Cruz
//!
//!
//! Based on ATMEL's development libraries
//! 
//! This file has been validated with AVRStudio-412462/WinAVR-20060125.
//! 
//******************************************************************************

//_____ I N C L U D E S ________________________________________________________
#include "config.h"
#include <uart_lib.h>
#include <uart_drv.h>
#include <sensor_drv.h>
#include <led_drv.h>
#include <hal.h>
#include <math.h>

#include "my_can.h"
#include "simple_rtc.h"

//#define STARTER_PERIOD_ms 50
 
//_____ D E F I N I T I O N S __________________________________________________

#define NODE_ID 1

#if NODE_ID == 1
	#define POS_X 1
	#define POS_Y 1
#elif NODE_ID == 2
	#define POS_X 50
	#define POS_Y 1
#elif NODE_ID == 3
	#define POS_X 100
	#define POS_Y 1
#elif NODE_ID == 4
	#define POS_X 1
	#define POS_Y 50
#elif NODE_ID == 5
	#define POS_X 50
	#define POS_Y 50
#elif NODE_ID == 6
	#define POS_X 100
	#define POS_Y 50
#elif NODE_ID == 7
	#define POS_X 1
	#define POS_Y 100
#elif NODE_ID == 8
	#define POS_X 50
	#define POS_Y 100
#elif NODE_ID == 9
	#define POS_X 100
	#define POS_Y 100
#endif

#define NUM_ITER 6

#define MAX_NODES 10 // these are the max nodes we can use
#define NPRIOBITS 27 // arbitration on CAN 2.0b
const U32 MAXV = (pow(2.0,NPRIOBITS)-1);

//_____ F U N C T I O N S ______________________________________________________

S8 send_and_rcv(U32 *prio, U32 value, U16 x, U16 y) 
{
	U8 u8_temp;
	packet_t pkt;
	
	// --- Enable Rx again
	//my_can_cmd_rx(&message_rx);

	//halWait_us(100);

	pkt.prio = *prio;
	pkt.value = value;
	pkt.x = x;
	pkt.y = y;

	if (my_can_send(TYPE_INTERPOLATION_DATA, &pkt)<0) {
		return -1;
	}

	halWait_us(1000);

	u8_temp = can_get_status(&message_rx);

	if (u8_temp == CAN_STATUS_COMPLETED && message_rx.id.ext != TYPE_START_INTERPOLATION_ITERATION) {
		my_can_cmd_rx(&message_rx);
		*prio = message_rx.id.ext;
	}

	u8_temp = can_get_status(&message_rx);

	return 0;
}	

U8 get_sensor_value()
{
	
	U8 lum=get_luminosity();
	write_led((1 << (lum/16))-1);
	//lum = NODE_ID*8;
	return lum;
}

//------------------------------------------------------------------------------
//  @fn main
//!
//! Core of "main_can_spy_echo_example.c".
//!
//! - CAN communications are listen, captured and a resume is sent on UART_0 and
//!   UART_1 (The UART number is at the end & at the beginning of each generated
//!   printing; i.e. 0 or 1).
//! - An CAN Echo also is sent: ID(sent)=ID(received)+1
//!                             DLC = 8
//!                             RTR = 0
//!                             Data[6:7]=CANSTMH:CANSTML
//! Note: Exit in UART ECHO MODE if CAN ID == 0 as CAN error simulation
//!
//! @warning UART's at 38400 bauds
//!          CAN in auto-bitrate
//!
//! @param none.
//!
//! @return Integer 0
//!
//------------------------------------------------------------------------------
int main (void)
{
#ifdef STARTER_PERIOD_ms
	U16 i;
	packet_t my_pkt;
#endif
	U8 u8_temp, iteration;
	U8 led_out = 0;
	U32 winner_prio;
	packet_t *rcv_packet;

	U16 x, y;
	U32 dx; 
	U32 dy; 
	U64 square;
	U64 value_64;
	U64 Big = 16777216; // 2^24

	U64 num=0; 
	U64 den=0; 
	U32 my_interpolated_value=0;

	U16 error;
	U32 prio, temp_prio, temp_prio_MAC_order;
	
	U8 num_interp_points = 0; //used to keep track of packets in S, goes up to 255 (U8)
	packet_t interpolation_set[NUM_ITER]; // records all winning records

	Bool update_myinterpolation = TRUE;

	U8 sensor_value, my_sensor_value;

//--- I N I T
    // Clock prescaler Reset
    CLKPR = 0x80;  CLKPR = 0x00;
    DDRA =0xFF;

    // --- Init LED's configuration:
    led_init(); write_led(led_out);

    // Init UART
    uart_init(CONF_8BIT_NOPAR_1STOP, UART_BAUDRATE); // UART_BAUDRATE defined in config.h
        
    uart_mini_printf ((U8 *)"\r\n\n ======== CAN TEST ======= (%d)\r\n", MSG_TX_TIME_clkTks);

    my_can_init();

	simple_rtc_init();

#ifdef STARTER_PERIOD_ms

	do {
		for (u8_temp=0; u8_temp<NUM_ITER; u8_temp++) {
			my_pkt.value = u8_temp; my_pkt.x = 0; my_pkt.y = 0;
			write_led(u8_temp+1);  //led_out++;

			if (my_can_send(TYPE_START_INTERPOLATION_ITERATION, &my_pkt)==0) {
				uart_mini_printf ((U8 *)"\r\n ===== CAN ERROR Tx1");
			} else { 
				uart_mini_printf ((U8 *)"\r\n *");
			}

			for (dx=0; dx<STARTER_PERIOD_ms; dx++) halWait_us(995);
		}
	} while (1);

#endif

	sensor_value = get_sensor_value();

    do {
    	//- WAIT FOR RECEIVE

		while ((CAN_PORT_IN & (1<<CAN_INPUT_PIN)) != 0) {
			simple_rtc_reset();
		}

		do {
        	// --- Wait for Rx completed
        	u8_temp = can_get_status(&message_rx);
		} while (u8_temp == CAN_STATUS_NOT_COMPLETED);

		if (u8_temp == CAN_STATUS_ERROR) {
			//handle error
			uart_mini_printf ((U8 *)"\r\n ===== CAN ERROR Rx!");
			my_can_init();
			//break;
		} else if (u8_temp == CAN_STATUS_COMPLETED) {

				// --- Enable Rx again
				my_can_cmd_rx(&message_rx);

				if ((message_rx.id.ext & 0x7) == TYPE_START_INTERPOLATION_ITERATION) {

					iteration = buffer_rx[0];
					
					if (iteration == 0) {
						//init iterpolation
						num_interp_points = 0; 
						my_interpolated_value = 0;
						my_sensor_value = get_sensor_value();
					}
						
					x = POS_X;
					y = POS_Y;
					update_myinterpolation = TRUE;
					sensor_value = my_sensor_value;

					if(sensor_value >= my_interpolated_value)
						error = sensor_value - my_interpolated_value;
					else
						error = my_interpolated_value - sensor_value;
				
					temp_prio = (U32)(((U32)error) * ((U32)(MAX_NODES+1)) + (U32)NODE_ID);
					temp_prio_MAC_order = ((U32)((1<<27)-1)) - temp_prio;
					
					prio = ((U32)(1<<27)) + temp_prio_MAC_order; 
					
					winner_prio = prio;
					//winner_prio = PRIO;

	    			//- SEND
					if (send_and_rcv(&winner_prio, sensor_value, POS_X, POS_Y)<0) {
						uart_mini_printf ((U8 *)"\r\n CAN ERROR Tx!");
						//break;
					}

					//write_led(winner_prio); 

					rcv_packet = (packet_t *) ((&buffer_rx[0])-SIZE_OF_PRIO);

					if(winner_prio == prio)
					{
						my_interpolated_value = sensor_value;
						update_myinterpolation = FALSE;
						//uart_mini_printf ((U8 *)"WON :");
					} else {
						//uart_mini_printf ((U8 *)"LOST :");
					}
					
					//uart_mini_printf ((U8 *)" <%lu, %lu, %u>\r\n", x, y, sensor_value);

					if(update_myinterpolation == TRUE)
					{
						dx = (U32)(x - rcv_packet->x);
						dy = (U32)(y - rcv_packet->y);

						square = (U64)((dx*dx) + (dy*dy));

						value_64 = (U64)(rcv_packet->value);

						num = num + ((value_64 * Big) / square);
						den = den + (Big / square);

						if(den == 0 || num == 0)
							my_interpolated_value = 0;
						else
							my_interpolated_value = (U32)(num / den);

						sensor_value = rcv_packet->value;
						x=rcv_packet->x;
						y=rcv_packet->y;
					}

					interpolation_set[num_interp_points].prio = winner_prio;
					interpolation_set[num_interp_points].value = sensor_value;
					interpolation_set[num_interp_points].x = x;
					interpolation_set[num_interp_points++].y = y;

#if NODE_ID == 1 // only node 1 ouputs stuff
					if (iteration == NUM_ITER - 1) {
						uart_mini_printf ( (U8*) "<%d, %d, %d;", interpolation_set[0].x, interpolation_set[0].y, interpolation_set[0].value+1);
						for (x=1; x<num_interp_points-1; x++) {
							uart_mini_printf ( (U8*) "%d, %d, %d;", interpolation_set[x].x, interpolation_set[x].y, interpolation_set[x].value+1);
						}
						uart_mini_printf ( (U8*) "%d, %d, %d>", interpolation_set[x].x, interpolation_set[x].y, interpolation_set[x].value+1);
					}
#endif
				}
        }

    } while(1);
    
//--- C A N    E R R O R
    uart_mini_printf ((U8 *)"\r\n ===== CAN ERROR ");

    return 0;
}
