/* ========================================================================== */
/*                                                                            */
/*   my_can.c																                                  */
/*   (c) 2007 Nuno Pereira                                                    */
/*                                                                            */
/*   Description                                                              */
/*          Basic can wrapper functions										                    */
/* ========================================================================== */
#include "config.h"
#include "my_can.h"
#include "can_lib.h"
#include "can_drv.h"
#include "hal.h"

st_cmd_t message_rx;
st_cmd_t message_tx;

U8 buffer_rx[8];

const U16 MIN_MSG_TX_TIME_us =		((U16)((double)MIN_MSG_TX_TIME_ms * 1000));
const U16 MSG_TX_TIME_clkTks =	((((U16)(((double)MSG_TX_TIME_ms * 1000)/CLOCK_TICK_TIME_us)) / 1000) * 1000);

void my_can_init() 
{
	U8 i;

    //- Pull-up on TxCAN & RxCAN one by one to use bit-addressing
    CAN_PORT_DIR &= ~(1<<CAN_INPUT_PIN );
    CAN_PORT_DIR &= ~(1<<CAN_OUTPUT_PIN);
    CAN_PORT_OUT |=  (1<<CAN_INPUT_PIN );
    CAN_PORT_OUT |=  (1<<CAN_OUTPUT_PIN);

	//CANGCON |= (1 << TTC);

	//- Reset CAN peripheral
    Can_reset();
 
	//CANGCON |= (1 << TTC);

    //- Set CAN Bit-timming
    can_init((U16)CAN_BAUDRATE);        // c.f. macro in "can_drv.h"

    //- Set CAN Timer Prescaler
    CANTCON = CANBT1;                   // Why not !

	CANGCON |= (1 << TTC);

	// --- Init Rx/Tx data
    message_rx.pt_data = &buffer_rx[0];
    for(i=0; i<8; i++) {
		buffer_rx[i]=0;
	}

    // --- Rx Command
    message_rx.cmd = CMD_RX;
        
    // --- Enable Rx
    while(my_can_cmd_rx(&message_rx) != CAN_CMD_ACCEPTED);
}

U8 my_can_cmd_prepare_tx(st_cmd_t* cmd) // accepts a st_cmd_t* to be compatible with can lib
{
	U8 cpt;

	cmd->cmd=CMD_TX_DATA;

	cmd->status = MOB_PENDING; 
    cmd->handle = MOB_0;
    Can_set_mob(cmd->handle);
    Can_clear_mob();
	
	if (cmd->ctrl.ide){ Can_set_ext_id(cmd->id.ext);}
	else              { Can_set_std_id(cmd->id.std);}
	for (cpt=0;cpt<cmd->dlc;cpt++) CANMSG = *(cmd->pt_data + cpt);
	if (cmd->ctrl.rtr) Can_set_rtr(); 
	else Can_clear_rtr();    
	Can_set_dlc(cmd->dlc);

	return CAN_CMD_ACCEPTED;
}
		

U8 my_can_cmd_rx(st_cmd_t* cmd) // accepts a st_cmd_t* to be compatible with can lib
{
  	U32 u32_temp;

	cmd->cmd=CMD_RX;

	cmd->status = MOB_PENDING; 
    cmd->handle = MOB_14;
    Can_set_mob(cmd->handle);
    Can_clear_mob();
	
	u32_temp=0; Can_set_ext_msk(u32_temp);
	Can_clear_rtrmsk();
	Can_clear_idemsk();
	Can_config_rx(); 

	return CAN_CMD_ACCEPTED;
}

S8 my_can_send(U8 type, packet_t *pkt) {

	U8 u8_temp;

    message_tx.pt_data = (U8 *) &(pkt->value);

    // --- Tx Command
	message_tx.ctrl.ide=1;
	if (type == TYPE_INTERPOLATION_DATA)
	{
		//message.id.ext= ((U32)(pkt->prio & 0x3FFFFFF) << 3) + type; 
		message_tx.id.ext = pkt->prio;
		message_tx.dlc = 8;
	}
	else // type = TYPE_START_INTERPOLATION_ITERATION
	{
		message_tx.id.ext= (U32) type; 
		message_tx.dlc = 2;
		*message_tx.pt_data = (U8)(pkt->value);
	}
    //message_tx.cmd = CMD_TX_DATA;
	        
	// --- Prepare Tx	
	my_can_cmd_prepare_tx(&message_tx);

    // --- Enable Tx at the right time
	simple_rtc_wait_until(MSG_TX_TIME_clkTks);
	Can_config_tx();

	u8_temp = can_get_status(&message_tx);

	if (type == TYPE_INTERPOLATION_DATA) {
		// --- Only allow for one transmission
		halWait_us(MIN_MSG_TX_TIME_us);
		//halWait_us(800);
      	Can_set_mob(message_tx.handle);
      	Can_mob_abort();
      	Can_clear_status_mob();
      	message_tx.handle = 0;		
	} else {
    	// --- Wait for Tx completed        
    	while(1)
    	{
        	u8_temp = can_get_status(&message_tx);
        	if (u8_temp != CAN_STATUS_NOT_COMPLETED) break; // Out of while
    	}

    	// ---- Exit if CAN error(s)
    	if (u8_temp == CAN_STATUS_ERROR) return -1; // Out of function
	}

	return 0;
}
